#include "MediaClock.hpp"

namespace twitch {
MediaClock::MediaClock()
    : m_playbackRate(1.0)
{
    reset();
}

void MediaClock::reset()
{
    m_currentTime = MediaTime::invalid();
    m_startTime = MediaTime::zero();
    m_wallClockStart = MediaTime::zero();
    m_wallClockElapsed = MediaTime::zero();
    m_started = false;
    m_renderTimes.clear();
}

MediaTime MediaClock::getElapsedMediaTime() const
{
    return m_started ? m_currentTime - m_startTime : MediaTime::zero();
}

MediaTime MediaClock::getElapsedWallClockTime() const
{
    if (m_wallClockStart == MediaTime::zero()) {
        return MediaTime::zero();
    } else {
        return ((now() - m_wallClockStart) * m_playbackRate) + m_wallClockElapsed;
    }
}

MediaTime MediaClock::getMediaTime() const
{
    return m_currentTime;
}

MediaTime MediaClock::getAudioTime() const
{
    for (const auto& entry : m_renderTimes) {
        if (entry.first.isAudio()) {
            return entry.second;
        }
    }
    return MediaTime::invalid();
}

MediaTime MediaClock::getVideoTime() const
{
    for (const auto& entry : m_renderTimes) {
        if (entry.first.isVideo()) {
            return entry.second;
        }
    }
    return MediaTime::invalid();
}

void MediaClock::setPlaybackRate(float rate)
{
    if (m_started) {
        m_wallClockElapsed = getElapsedWallClockTime();
        m_wallClockStart = now();
    }
    m_playbackRate = rate;
}

void MediaClock::setDiscontinuity(bool discontinuity)
{
    if (discontinuity) {
        for (auto& entry : m_renderTimes) {
            entry.second = MediaTime::invalid();
        }
    }
}

void MediaClock::setTime(MediaTime time)
{
    m_currentTime = time;
    if (m_startTime == MediaTime::zero()) {
        m_startTime = time;
    }
}

void MediaClock::setTime(const MediaType& type, MediaTime time)
{
    m_renderTimes[type] = time;
}

MediaTime MediaClock::now() const
{
    return MediaTime::now();
}

void MediaClock::start()
{
    if (!m_started) {
        m_started = true;
        m_wallClockStart = now();
    }
}

void MediaClock::stop()
{
    if (m_started) {
        m_wallClockElapsed = getElapsedWallClockTime();
        m_started = false;
    }
}
}
