#include "PlaybackSink.hpp"

namespace twitch {
PlaybackSink::PlaybackSink(NativePlatform& platform, MediaSink::Listener& listener, const std::shared_ptr<Scheduler>& scheduler)
    : ScopedScheduler(scheduler)
    , m_platform(platform)
    , m_listener(listener)
    , m_log(platform.getLog())
    , m_scheduler(scheduler)
    , m_syncMediaType(MediaType::Type_Audio, "*")
    , m_volume(1.0)
    , m_surface(nullptr)
    , m_paused(true)
{
}

PlaybackSink::~PlaybackSink()
{
    m_paused = true;
    // make sure no callbacks run by canceling all of them
    ScopedScheduler::cancel();
    m_tracks.clear(); // safe the sink is deleted on the player's scheduler thread
}

void PlaybackSink::configure(int track, std::shared_ptr<MediaFormat> format)
{
    // use a wildcard media type, this keeps video/audio tracks consistent across different formats
    // and uses just one text track
    MediaType type(format->getType().type, "*");

    // check content protection is supported by the platform
    if (format && !format->getProtectionData().empty() && !checkProtectionSupported(*format)) {
        onTrackError(type, Error(ErrorSource::Decode, MediaResult::ErrorNotSupported, "Protection system not supported"));
        return;
    }

    if (m_tracks.count(type) == 0) {
        m_tracks[type] = createTrack(format);

        if (!m_paused) {
            m_log->info("track %d - %s added after already playing", track, type.name.c_str());
            m_tracks[type]->play();
        }
    }

    m_trackTypes[track] = type;
    m_tracks[type]->configure(format);
}

void PlaybackSink::enqueue(int track, std::shared_ptr<MediaSampleBuffer> sample)
{
    const MediaType& type = m_trackTypes[track];
    auto it = m_tracks.find(type);

    if (it != m_tracks.end() && it->second) {
        it->second->enqueue(sample);
    } else {
        m_log->error("No sink for track %d", track);
    }
}

void PlaybackSink::prepare()
{
    // if the content is protected this will be deferred until the keys have been provided to
    // the platform DRM module, otherwise this will remain true
    if (!m_drmClient || m_drmClient->isKeyUpdated()) {
        for (const auto& entry : m_tracks) {
            entry.second->prepare();
        }
    }
}

void PlaybackSink::play()
{
    if (m_paused) {
        m_paused = false;
        prepare();
    }
}

void PlaybackSink::pause()
{
    for (const auto& entry : m_tracks) {
        entry.second->pause();
    }

    m_paused = true;
}

void PlaybackSink::remove(const TimeRange& range)
{
    for (const auto& entry : m_tracks) {
        entry.second->remove(range);
    }
}

void PlaybackSink::reset()
{
    for (const auto& entry : m_tracks) {
        entry.second->flush();
    }

    std::unique_lock<Mutex> lock(m_mutex);
    m_clock.reset();
    m_prepared.clear();
    m_drmClient.reset();
    m_trackTypes.clear();
}

void PlaybackSink::seekTo(MediaTime time)
{
    // prepare tracks first
    for (const auto& entry : m_tracks) {
        m_prepared[entry.first] = false;
    }
    for (const auto& entry : m_tracks) {
        entry.second->seekTo(time);
    }

    std::unique_lock<Mutex> lock(m_mutex);
    m_clock.reset();
    m_clock.setTime(time);
}

void PlaybackSink::setPlaybackRate(float rate)
{
    for (const auto& entry : m_tracks) {
        entry.second->setPlaybackRate(rate);
        if (m_syncMediaType.matches(entry.first)) {
            m_clock.setPlaybackRate(rate);
        }
    }
}

void PlaybackSink::setSurface(void* surface)
{
    m_surface = surface;
    for (const auto& entry : m_tracks) {
        if (entry.first.isVideo()) {
            entry.second->setSurface(surface);
        }
    }
}

void PlaybackSink::setVolume(float volume)
{
    m_volume = volume;
    for (const auto& entry : m_tracks) {
        if (entry.first.isAudio()) {
            entry.second->setVolume(m_volume);
        }
    }
}

bool PlaybackSink::checkProtectionSupported(const MediaFormat& format)
{
    if (!m_drmClient) {
        m_drmClient.reset(new DrmClient(m_platform, *this, m_scheduler));
    }
    return m_drmClient->onProtectedMedia(format);
}

std::unique_ptr<TrackSink> PlaybackSink::createTrack(const std::shared_ptr<MediaFormat>& format)
{
    bool isVideo = format->getType().isVideo();
    bool isAudio = format->getType().isAudio();
    auto track = std::unique_ptr<TrackSink>(new TrackSink(m_platform, *this, m_clock, format));
    // set initial playback settings
    track->setPlaybackRate(m_clock.getPlaybackRate());
    if (isVideo) {
        track->setSurface(m_surface);
    }
    if (isAudio) {
        track->setVolume(m_volume);
    }
    MediaTime mediaTime = m_clock.getMediaTime();
    if (mediaTime.valid()) {
        track->seekTo(mediaTime);
    }
    return track;
}

void PlaybackSink::updateSyncTrack()
{
    for (const auto& entry : m_trackTypes) {
        if (entry.second.isAudio()) {
            m_syncMediaType = entry.second;
            return;
        }
    }
    // else prefer the video track
    for (const auto& entry : m_trackTypes) {
        if (entry.second.isVideo()) {
            m_syncMediaType = entry.second;
        }
    }
}

void PlaybackSink::onKeysProvided()
{
    if (!m_paused) {
        prepare();
    }
}

void PlaybackSink::onProtectionError(const Error& error)
{
    schedule([=]() {
        m_listener.onSinkError(error);
    });
}

void PlaybackSink::onTrackError(const MediaType& type, const Error& error)
{
    (void)type;
    schedule([=]() {
        m_listener.onSinkError(error);
    });
}

void PlaybackSink::onTrackConfigured(std::shared_ptr<const MediaFormat> format)
{
    if (format) {
        schedule([=]() {
            m_listener.onSinkFormatChanged(*format);
        });
    }
}

void PlaybackSink::onTrackMetadataSample(std::shared_ptr<const MediaSampleBuffer> sample)
{
    schedule([=]() {
        m_listener.onSinkMetadataSample(*sample);
    });
}

void PlaybackSink::onTrackPrepared(const MediaType& type)
{
    m_log->info("prepared %s", type.name.c_str());
    // this maybe called on the track's thread or the player scheduler thread
    schedule([=]() {
        m_prepared[type] = true;
        if (m_paused) {
            return;
        }
        // check all tracks are in prepared state
        for (const auto& entry : m_prepared) {
            auto& track = m_tracks[entry.first];
            if (!entry.second && track && !track->isEmpty()) {
                return;
            }
        }

        updateSyncTrack();
        for (const auto& entry : m_tracks) { // play all
            entry.second->play();
        }

        m_listener.onSinkStateChanged(MediaSink::State::Playing);
    });
}

void PlaybackSink::onTrackRecoverableError(const Error& error)
{
    schedule([this, error]() {
        m_listener.onSinkRecoverableError(error);
    });
}

void PlaybackSink::onTrackTimeDiscontinuity(const MediaType& type, MediaTime time)
{
    (void)time;
    if (!type.isText()) {
        // reset on both AV tracks when there is a discontinuity for now
        std::unique_lock<Mutex> lock(m_mutex);
        m_clock.reset();
        m_clock.setDiscontinuity(true);
    }
}

void PlaybackSink::onTrackTimeSkip(const MediaType& type, MediaTime time)
{
    schedule([this, type, time]() {
        // one track e.g. video may have invalid data before it can play if so discard data
        // from other tracks to keep in sync
        for (const auto& entry : m_tracks) {
            if (!type.matches(entry.first)) { // exclude the track that called this method
                entry.second->remove(TimeRange(MediaTime::zero(), time));
                break;
            }
        }
    });
}

void PlaybackSink::onTrackTimeUpdate(const MediaType& type, MediaTime time)
{
    if (m_syncMediaType.matches(type) && !m_paused) {
        m_clock.start();
        m_clock.setTime(time);
        schedule([=]() {
            m_listener.onSinkTimeUpdate(time);
        });
    }
    std::unique_lock<Mutex> lock(m_mutex);
    m_clock.setTime(type, time);
}

void PlaybackSink::onTrackIdle(const MediaType& type)
{
    if (m_syncMediaType.matches(type)) {
        m_clock.stop();
    }

    schedule([this]() {
        if (!m_paused) {
            // all tracks must in the idle state
            for (const auto& entry : m_tracks) {
                const auto& trackType = entry.first;
                if ((trackType.isVideo() || trackType.isAudio()) && !entry.second->isIdle()) {
                    return;
                }
            }
            m_listener.onSinkStateChanged(MediaSink::State::Idle);
        }
    });
}

void PlaybackSink::onTrackStatistics(const MediaType& type, const Statistics& statistics)
{
    if (type.isVideo()) {
        m_listener.onSinkVideoStatistics(statistics); // not scheduled
    }

    if (!m_paused) {
        checkClockSync();
    }
}

void PlaybackSink::checkClockSync()
{
    const MediaTime MaxDrift(1.0);
    // check audio/video time difference
    MediaTime audio;
    MediaTime video;
    {
        std::unique_lock<Mutex> lock(m_mutex);
        audio = m_clock.getAudioTime();
        video = m_clock.getVideoTime();
    }

    if (audio.valid() && audio > MediaTime::zero() && video.valid() && video > MediaTime::zero()) {
        // check A/V sync times
        if (m_syncMediaType.isAudio()) {
            MediaTime delta = audio - video;
            if (delta.absolute() > MaxDrift) {
                m_log->warn("Video time %lld us drifted %lld us from Audio time us %lld",
                    video.microseconds().count(), delta.microseconds().count(), audio.microseconds().count());
            }
        }
        // check wall clock drift
        MediaTime wall = m_clock.getMediaStartTime() + m_clock.getElapsedWallClockTime();
        if (wall != MediaTime::zero()) {
            MediaTime media = audio;
            if (m_syncMediaType.isVideo()) {
                media = video;
            }
            MediaTime delta = wall - media;
            if (delta.absolute() > MaxDrift) {
                m_log->warn("Wall clock time %lld us drifted %lld us from media time us %lld",
                    wall.microseconds().count(), delta.microseconds().count(), media.microseconds().count());
            }
        }
    }
}
}
