#pragma once

#include "DrmClient.hpp"
#include "MediaClock.hpp"
#include "TrackSink.hpp"
#include "player/ScopedScheduler.hpp"
#include "playercore/MediaSink.hpp"
#include "playercore/platform/NativePlatform.hpp"
#include "util/Concurrency.hpp"
#include <map>
#include <vector>

namespace twitch {
class PlaybackSink : public MediaSink,
                     private ScopedScheduler,
                     private TrackSink::Listener,
                     private DrmClient::Listener {
public:
    PlaybackSink(NativePlatform& platform, MediaSink::Listener& listener, const std::shared_ptr<Scheduler>& scheduler);
    ~PlaybackSink() override;
    PlaybackSink(const PlaybackSink&) = delete;
    const PlaybackSink& operator=(const PlaybackSink&) = delete;

    void configure(int track, std::shared_ptr<MediaFormat> format) override;
    void endOfStream() override {}
    void enqueue(int track, std::shared_ptr<MediaSampleBuffer> sample) override;
    void flush() override {};
    void play() override;
    void prepare() override;
    void pause() override;
    void remove(const TimeRange& range) override;
    void reset() override;
    void seekTo(MediaTime time) override;
    void setPlaybackRate(float rate) override;
    void setSurface(void* surface) override;
    void setVolume(float volume) override;

private:
    bool checkProtectionSupported(const MediaFormat& format);
    void checkClockSync();
    std::unique_ptr<TrackSink> createTrack(const std::shared_ptr<MediaFormat>& format);
    void updateSyncTrack();
    void onKeysProvided() override;
    void onProtectionError(const Error& error) override;
    void onTrackError(const MediaType& type, const Error& error) override;
    void onTrackConfigured(std::shared_ptr<const MediaFormat> format) override;
    void onTrackMetadataSample(std::shared_ptr<const MediaSampleBuffer> sample) override;
    void onTrackPrepared(const MediaType& type) override;
    void onTrackRecoverableError(const Error& error) override;
    void onTrackTimeSkip(const MediaType& type, MediaTime time) override;
    void onTrackTimeDiscontinuity(const MediaType& type, MediaTime time) override;
    void onTrackTimeUpdate(const MediaType& type, MediaTime time) override;
    void onTrackIdle(const MediaType& type) override;
    void onTrackStatistics(const MediaType& type, const Statistics& statistics) override;

    struct MediaTypeComparator {
        bool operator()(const MediaType& a, const MediaType& b) const { return !a.matches(b) && a < b; }
    };

    NativePlatform& m_platform;
    MediaSink::Listener& m_listener;
    MediaClock m_clock;
    Mutex m_mutex;
    std::shared_ptr<Log> m_log;
    std::shared_ptr<Scheduler> m_scheduler;
    std::map<MediaType, std::unique_ptr<TrackSink>, MediaTypeComparator> m_tracks;
    std::unique_ptr<DrmClient> m_drmClient;
    std::map<int, MediaType> m_trackTypes;
    std::map<MediaType, bool> m_prepared;
    MediaType m_syncMediaType;
    float m_volume;
    void* m_surface;
    bool m_paused;
};
}
