#pragma once

#include "Log.hpp"
#include "playercore/MediaTime.hpp"
#include "playercore/platform/NativePlatform.hpp"
#include "playercore/platform/Scheduler.hpp"
#include "util/Concurrency.hpp"
#include <atomic>
#include <map>
#include <queue>
#include <string>
#include <thread>
#include <vector>

namespace twitch {
class ThreadScheduler : public Scheduler, public std::enable_shared_from_this<ThreadScheduler> {
public:
    ThreadScheduler(NativePlatform& platform, const std::string& name, int threads = 1);
    ~ThreadScheduler() override;
    ThreadScheduler(const ThreadScheduler&) = delete;
    const ThreadScheduler& operator=(const ThreadScheduler&) = delete;

    std::shared_ptr<Cancellable> schedule(Action action, Microseconds time, bool repeating) override;
    void scheduleAndWait(Action action) override;

private:
    struct Task : public Cancellable, public std::enable_shared_from_this<Task> {
        Action action;
        MediaTime when = MediaTime::zero();
        bool cancelled = false;
        bool complete = false;
        bool repeating = false;
        MediaTime interval = MediaTime::zero();
        std::weak_ptr<ThreadScheduler> owner;
        std::thread::id thread = std::thread::id();

        bool operator<(const Task& other) const { return when < other.when; }
        bool operator>(const Task& other) const { return when > other.when; }
        void cancel() override;
        virtual void run();
    };

    void cancel(std::shared_ptr<Task> task);
    void processQueue();

    struct TaskComparator {
        bool operator()(const std::shared_ptr<Task>& a, const std::shared_ptr<Task>& b) { return *a > *b; }
    };
    struct Queue : std::priority_queue<std::shared_ptr<Task>, std::vector<std::shared_ptr<Task>>, TaskComparator> {
        bool remove(const std::shared_ptr<Task>& task);
    };

    Queue m_queue;
    NativePlatform& m_platform;
    std::string m_name;
    std::atomic<bool> m_run;
    Mutex m_mutex;
    ConditionVariable m_queueAvailable;
    ConditionVariable m_waitCondition;
    std::map<std::thread::id, std::shared_ptr<Task>> m_waitTasks;
    const int m_threadPoolSize;
    Mutex m_threadMutex;
    ConditionVariable m_threadCondition;
    std::vector<std::thread> m_threads;
    int m_threadsRunning;
};
}
