#include "TrackBuffer.hpp"
#include <algorithm>

namespace twitch {
TrackBuffer::TrackBuffer(const Platform& platform, const MediaType& mediaType)
    : m_log(platform.getLog(), "Media " + mediaType.type + " : ")
    , m_isVideo(mediaType.isVideo())
    , m_reorderSamples(mediaType.isText())
    , m_checkOverlap(false)
{
}

void TrackBuffer::push(const std::shared_ptr<const MediaFormat>& format)
{
    // handle case where tracks configured multiple times but no sample data is added
    if (!m_queue.empty()) {
        auto& message = m_queue.back();
        if (!message.sample && message.format) {
            message.format = format;
            return;
        }
    }

    m_queue.emplace_back(format);
}

void TrackBuffer::push(const std::shared_ptr<MediaSampleBuffer>& sample)
{
    // checks for out of order samples after a remove() call
    if (m_checkOverlap) {
        removeBack(sample->decodeTime);
        m_checkOverlap = false;
    }

    if (!m_queue.empty()) {
        auto& message = m_queue.back();
        if (message.format && !message.sample) {
            message.sample = sample;
        } else {
            // on the metadata track sample pts can be out of order handle it by finding
            // the right insertion point for the sample
            if (m_reorderSamples && !sample->isDiscontinuity) {
                if (message.sample && sample->presentationTime < message.sample->presentationTime) {
                    auto it = m_queue.rbegin();
                    for (; it != m_queue.rend(); it++) {
                        if (it->sample && sample->presentationTime > it->sample->presentationTime) {
                            break;
                        }
                    }
                    m_queue.insert(it.base(), Item(sample));
                    return;
                }
            }
            m_queue.emplace_back(sample);
        }
    } else {
        m_queue.emplace_back(sample);
    }
}

bool TrackBuffer::hasPresentationTime(MediaTime time)
{
    for (const auto& message : m_queue) {
        if (message.sample && message.sample->presentationTime >= time) {
            return true;
        }
    }
    return false;
}

void TrackBuffer::removeBack(MediaTime time)
{
    while (!m_queue.empty()) {
        if (m_queue.back().sample && m_queue.back().sample->decodeTime > time) {
            m_log.info("back sample ahead of next sample %lld us > %lld us",
                m_queue.back().sample->decodeTime.microseconds().count(),
                time.microseconds().count());
            m_queue.pop_back();
        } else {
            break;
        }
    }
}

void TrackBuffer::remove(const TimeRange& range)
{
    if (range.duration > MediaTime::zero()) {
        remove(range, false);
        // check overlapping samples for newly enqueued samples
        m_checkOverlap = true;
    }
}

void TrackBuffer::remove(const TimeRange& range, bool seeking)
{
    // find the furthest time we can discard to (exclusive)
    auto discardStart = m_queue.end();
    auto discardEnd = m_queue.begin();
    std::shared_ptr<const MediaFormat> format;

    for (auto it = m_queue.begin(); it < m_queue.end(); ++it) {
        auto& message = *it;

        // keep track of the media format when seeking
        if (message.format) {
            format = message.format;
        }

        if (message.sample && message.sample->decodeTime >= range.start && it < discardStart) {
            discardStart = it;
        }

        if (message.sample && message.sample->decodeTime >= range.end()) {
            break;
        }

        if (m_isVideo && seeking) {
            if (message.sample && message.sample->isSyncSample && it > m_queue.begin()) {
                discardEnd = it;
            }
        } else {
            discardEnd = it;
        }
    }

    if (discardEnd > discardStart) {
        auto size = m_queue.size();
        m_queue.erase(discardStart, discardEnd);
        auto remaining = m_queue.size();
        // edge case if only 1 sample remaining drop it
        if (remaining == 1) {
            m_queue.clear();
            remaining = 0;
        }

        // set the format to the front of the queue if needed
        if (format && !m_queue.empty() && !m_queue.front().format) {
            m_queue.front().format = format;
        }

        auto dropped = size - remaining;
        if (dropped) {
            m_log.info("removed %d samples (remaining %d)", dropped, remaining);
        }
    }
}

MediaTime TrackBuffer::removeToSyncFrame()
{
    while (!m_queue.empty()) {
        const auto& message = m_queue.front();
        if (message.sample && message.sample->isSyncSample) {
            return message.sample->decodeTime;
        }
        m_queue.pop_front();
    }

    return MediaTime::zero();
}

void TrackBuffer::seek(MediaTime seekTime)
{
    remove(TimeRange(MediaTime::zero(), seekTime), true);
    // mark the next sample in the pipeline as a isDiscontinuity to flush the renderer
    if (!m_queue.empty() && m_queue.front().sample) {
        auto sample = m_queue.front().sample;
        if (m_isVideo) {
            if (sample->isSyncSample) {
                sample->isDiscontinuity = true;
            }
        } else {
            sample->isDiscontinuity = true;
        }
    }
}
}
