#include "TrackRenderer.hpp"
#include "MediaResult.hpp"
#include "NullAudioRenderer.hpp"
#include "NullVideoRenderer.hpp"
#include "PassthroughDecoder.hpp"

namespace twitch {
TrackRenderer::TrackRenderer(NativePlatform& platform,
    CancellationFlag& flag,
    TrackRenderer::Listener& listener,
    MediaClock& clock,
    const std::shared_ptr<const MediaFormat>& format)
    : m_platform(platform)
    , m_log(platform.getLog(), "Track " + format->getType().type + " : ")
    , m_flag(flag)
    , m_listener(listener)
    , m_clock(clock)
    , m_decoder(m_platform.createDecoder(format))
    , m_renderer(m_platform.createRenderer(clock, format))
    , m_decodedTime(MediaTime::invalid())
    , m_renderedTime(MediaTime::invalid())
    , m_decoderFormat(format)
    , m_configured(false)
    , m_rendererStarted(false)
{
    const auto& mediaType = m_decoderFormat->getType();
    // create passthrough decoder if none provided
    if (!m_decoder) {
        m_decoder.reset(new PassthroughDecoder());
    }
    // create null renderer if none provided
    if (!m_renderer) {
        if (mediaType.isText()) {
            // create metadata track to fire the metadata at the correct time
            m_renderer.reset(new MetadataRenderer(listener, m_clock));
        } else if (mediaType.isAudio()) {
            m_renderer.reset(new NullAudioRenderer());
        } else {
            m_renderer.reset(new NullVideoRenderer());
        }
    }
    m_log.info("created");
}

TrackRenderer::~TrackRenderer()
{
    m_log.info("destroying");
}

void TrackRenderer::configure(const std::shared_ptr<const MediaFormat>& format, bool discontinuous)
{
    if (discontinuous) {
        m_decodedTime = MediaTime::invalid();
        m_renderedTime = MediaTime::invalid();
    }

    // check if the decoder type is changing
    if (!m_decoderFormat->getType().matches(format->getType())) {
        m_log.info("Decoder type change %s to %s",
            m_decoderFormat->getType().name.c_str(), format->getType().name.c_str());
        // create new decoder for type
        m_decoder = m_platform.createDecoder(format);
        if (!m_decoder) {
            m_decoder.reset(new PassthroughDecoder());
        }
    }

    // flush the decoder output if started previously
    if (m_configured) {
        m_log.info("reconfigure discontinuous %s", discontinuous ? "true" : "false");

        if (discontinuous) {
            if (decodeOk(m_decoder->reset(), "Decoder reset failed")) {
                renderOk(m_renderer->flush(), "Renderer flush failed");
            }
        } else if (decodeOk(m_decoder->flush(), "Decoder flush failed")) {
            // disabled on the text track which uses onTimedWait
            // TODO make this not happen during the lock phase in TrackSink
            if (!m_decoderFormat->getType().isText()) {
                processOutput();
            }
        }
    }

    // reconfigure decoder/renderer
    media::SourceFormat rendererFormat;
    m_configured = decodeOk(m_decoder->configure(*format, rendererFormat), "Decoder configure failed");
    m_decoderFormat = format;

    // only reconfigure the renderer if the decoder output format changes
    if (rendererFormat != m_rendererFormat) {
        m_rendererFormat = rendererFormat;
        if (!renderOk(m_renderer->configure(rendererFormat), "Renderer configure failed")) {
            return;
        }
    }
}

void TrackRenderer::flush()
{
    if (isConfigured()) {
        renderOk(m_renderer->flush(), "Renderer flush failed");
    }
}

bool TrackRenderer::processInput(const std::shared_ptr<const MediaSampleBuffer>& sample)
{
    MediaResult result = m_decoder->decode(*sample);

    if (result == MediaResult::ErrorTimeout || !decodeOk(result, "Failed to decode sample")) {
        return false;
    }

    m_decodedTime = sample->decodeTime;
    if (!sample->isDecodeOnly) {
        updateStatistics(*sample);
    }

    return true;
}

void TrackRenderer::processOutput()
{
    std::shared_ptr<MediaSample> sample = std::make_shared<MediaSampleBuffer>();

    while (!m_flag.isCancelled()) {
        bool hasOutput = false;
        if (decodeOk(m_decoder->hasOutput(hasOutput), "Decoder hasOutput failed")) {
            if (!hasOutput) {
                break;
            }

            if (decodeOk(m_decoder->getOutput(sample), "Decoder getOutput failed")) {
                if (sample) {
                    if (renderOk(m_renderer->render(sample), "Render sample failed")) {
                        m_statistics.renderedFrame();
                        updateRendererTime(); // update the rendered presentation time
                    }
                }
            } else {
                return;
            }
        } else {
            return;
        }
    }

    // update the rendered presentation time (if no decoder output)
    updateRendererTime();
}

void TrackRenderer::updateState(bool idle, bool paused)
{
    if (idle) {
        if (m_rendererStarted && m_renderer) {
            if (!paused) {
                waitForRendererIdle();
            }

            if (!renderOk(m_renderer->stop(), "Renderer stop failed")) {
                return;
            }
        }

        if (!m_decoderFormat->getType().isText() && m_renderedTime.valid()) {
            m_log.info("idle at %lld us", m_renderedTime.microseconds().count());
        }

    } else {
        m_rendererStarted = renderOk(m_renderer->start(), "Renderer start failed");
    }
}

void TrackRenderer::waitForRendererIdle()
{
    while (!m_flag.isCancelled() && m_decodedTime.valid() && m_renderedTime.valid() && (m_decodedTime > m_renderedTime)) {
        processOutput();
        MediaTime lastRenderTime = m_renderedTime;
        MediaTime remaining = m_decodedTime - m_renderedTime;
        if (!m_listener.onTimedWait(remaining)) {
            return;
        }

        updateRendererTime();

        if (m_renderedTime == lastRenderTime) {
            return; // time is not advancing
        }
    }
}

void TrackRenderer::updateStatistics(const MediaSampleBuffer& sample)
{
    MediaTime end = m_clock.now();

    if (m_statistics.update(sample, end)) {
        if (m_renderer && m_decoderFormat->getType().isVideo()) {
            VideoRenderer* videoRenderer = static_cast<VideoRenderer*>(m_renderer.get());
            int droppedFrames = 0;
            videoRenderer->getDroppedFrames(droppedFrames);
            m_statistics.setDroppedFrames(droppedFrames);

            int renderedFrames = 0;
            videoRenderer->getRenderedFrames(renderedFrames);
            m_statistics.setRenderedFrames(renderedFrames);
        }
        m_listener.onStatisticsUpdated(m_statistics);
    }
}

void TrackRenderer::setPlaybackRate(float rate)
{
    MediaResult result = m_renderer->setPlaybackRate(rate);
    if (result == MediaResult::ErrorNotSupported) { // not supported on all platforms yet
        m_log.warn("%s - %s", mediaResultString(result), "Playback rate not supported");
    } else {
        renderOk(result, "Error setting playback rate");
    }
}

void TrackRenderer::setSurface(void* surface)
{
    if (m_decoderFormat->getType().isVideo()) {
        VideoRenderer* videoRenderer = static_cast<VideoRenderer*>(m_renderer.get());
        renderOk(videoRenderer->setSurface(surface), "Error setting surface");
    }
}

void TrackRenderer::setVolume(float volume)
{
    if (m_decoderFormat->getType().isAudio()) {
        AudioRenderer* audioRender = static_cast<AudioRenderer*>(m_renderer.get());
        renderOk(audioRender->setVolume(volume), "Error setting volume");
    }
}

void TrackRenderer::updateRendererTime()
{
    // update rendered presentation time
    MediaTime renderTime = MediaTime::invalid();

    if (renderOk(m_renderer->getRenderedPresentationTime(renderTime), "Failed to get renderer presentation time")) {
        if (renderTime.valid()) {
            m_renderedTime = renderTime;
            m_listener.onRenderTimeUpdate(m_renderedTime);
        }
    }
}

bool TrackRenderer::decodeOk(MediaResult result, const std::string& message)
{
    if (result != MediaResult::Ok) {
        m_listener.onDecodeError(result, message);
        return false;
    }
    return true;
}

bool TrackRenderer::renderOk(MediaResult result, const std::string& message)
{
    if (result != MediaResult::Ok) {
        m_listener.onRenderError(result, message);
        return false;
    }
    return true;
}
}
