#pragma once

#include "CancellationFlag.hpp"
#include "MediaClock.hpp"
#include "MetadataRenderer.hpp"
#include "TrackStatistics.hpp"
#include "debug/PrefixLog.hpp"
#include "media/SourceFormat.hpp"
#include "playercore/platform/NativePlatform.hpp"
#include <string>

namespace twitch {
class TrackRenderer {
public:
    class Listener : public MetadataRenderer::Listener {
    public:
        ~Listener() override = default;
        virtual void onDecodeError(MediaResult result, const std::string& message) = 0;
        virtual void onRenderError(MediaResult result, const std::string& message) = 0;
        virtual void onRenderTimeUpdate(MediaTime time) = 0;
        virtual void onStatisticsUpdated(const Statistics& statistics) = 0;
    };

    TrackRenderer(NativePlatform& platform,
        CancellationFlag& flag,
        Listener& listener,
        MediaClock& clock,
        const std::shared_ptr<const MediaFormat>& format);
    virtual ~TrackRenderer();
    TrackRenderer(const TrackRenderer&) = delete;
    const TrackRenderer& operator=(const TrackRenderer&) = delete;

    void configure(const std::shared_ptr<const MediaFormat>& format, bool discontinuous);
    void flush();
    bool processInput(const std::shared_ptr<const MediaSampleBuffer>& sample);
    void processOutput();
    void updateState(bool idle, bool paused);
    bool isConfigured() { return m_configured; }
    MediaTime getRenderedTime() { return m_renderedTime; }
    void setPlaybackRate(float rate);
    void setSurface(void* surface);
    void setVolume(float volume);

private:
    void updateRendererTime();
    void updateStatistics(const MediaSampleBuffer& sample);
    void waitForRendererIdle();
    bool decodeOk(MediaResult result, const std::string& message);
    bool renderOk(MediaResult result, const std::string& message);

    NativePlatform& m_platform;
    PrefixedLog m_log;
    CancellationFlag& m_flag;
    TrackRenderer::Listener& m_listener;
    MediaClock& m_clock;
    std::unique_ptr<MediaDecoder> m_decoder;
    std::unique_ptr<MediaRenderer> m_renderer;
    TrackStatistics m_statistics;
    MediaTime m_decodedTime;
    MediaTime m_renderedTime;
    std::shared_ptr<const MediaFormat> m_decoderFormat;
    media::SourceFormat m_rendererFormat;
    bool m_configured;
    bool m_rendererStarted;
};
}
