#include "TrackSink.hpp"
#include <algorithm>

namespace twitch {
TrackSink::TrackSink(NativePlatform& platform, Listener& listener, MediaClock& clock, const std::shared_ptr<MediaFormat>& format)
    : m_platform(platform)
    , m_log(platform.getLog(), "Track " + format->getType().type + " : ")
    , m_listener(listener)
    , m_clock(clock)
    , m_scheduler(platform.createScheduler("Player Track " + format->getType().type))
    , m_format(format)
    , m_mediaType(format->getType().type, "*")
    , m_buffer(platform, m_mediaType)
    , m_seekTime(MediaTime::zero())
    , m_paused(true)
    , m_idle(true)
    , m_seeking(false)
    , m_isVideo(format->getType().isVideo())
    , m_surface(nullptr)
{
    m_scheduler->schedule(std::bind(&TrackSink::processQueue, this));
}

TrackSink::~TrackSink()
{
    {
        std::lock_guard<Mutex> lock(m_mutex);
        m_run.cancel();
    }

    m_queueCondition.notify_all();
    m_waitCondition.notify_all();
    m_idleCondition.notify_all();
    m_surfaceChanged.notify_all();

    m_scheduler.reset();
}

void TrackSink::processQueue()
{
    bool skipConfigure = false;

    // scoped to the lifetime of the thread for thread safety, do not store as a member
    TrackRenderer renderer(m_platform, m_run, *this, m_clock, m_format);

    while (!m_run.isCancelled()) {
        bool isLastSample = false;
        MediaTime lastPresentationTime = MediaTime::zero();

        {
            std::unique_lock<Mutex> lock(m_mutex);

            while ((m_buffer.isEmpty() || m_paused) && !m_run.isCancelled()) {
                lock.unlock();
                dequeueActions(renderer);
                updateIdleState(renderer, true);
                skipConfigure = false;
                lock.lock();
                if ((m_buffer.isEmpty() || m_paused) && !m_run.isCancelled()) {
                    m_queueCondition.wait(lock);
                } else {
                    break;
                }
            }

            if (m_run.isCancelled()) {
                break;
            }

            TrackBuffer::Item message = m_buffer.front();
            if (!message.sample) {
                continue; // no sample to process
            }

            isLastSample = m_buffer.size() == 1;
            std::shared_ptr<MediaSampleBuffer> sample = message.sample;
            // if before the seek time don't render this sample (unless its the last frame of video)
            sample->isDecodeOnly = sample->presentationTime < m_seekTime && (!isLastSample || !m_isVideo);

            // reconfigure
            if (!skipConfigure && (!renderer.isConfigured() || sample->isDiscontinuity || message.format)) {
                if (!rendererConfigure(renderer, message.format, sample)) {
                    continue;
                }
            }

            if (renderer.processInput(sample)) { // returns true to continue processing input
                lastPresentationTime = sample->presentationTime;
                m_buffer.pop();
                skipConfigure = false;
            } else {
                skipConfigure = true;
            }
        }

        dequeueActions(renderer);

        if (!m_seeking || !m_isVideo) {
            updateIdleState(renderer, false);
        }

        renderer.processOutput();

        if (m_seeking) {
            // while seeking the video track is rendering the last frame to display so use the track rendered time
            auto renderedTime = m_isVideo ? renderer.getRenderedTime() : lastPresentationTime;

            if ((renderedTime.valid() && renderedTime >= m_seekTime) || isLastSample) {
                onSeekCompleted();
            }
        }
    }
}

bool TrackSink::rendererConfigure(TrackRenderer& renderer,
    const std::shared_ptr<const MediaFormat>& format,
    const std::shared_ptr<const MediaSampleBuffer>& sample)
{
    // wait for a key frame if we have undecodeable video
    if (m_isVideo && !sample->isSyncSample) {
        MediaTime startTime = m_buffer.removeToSyncFrame();
        if (startTime > MediaTime::zero()) {
            m_log.warn("skipped to %lld us", startTime.microseconds().count());
            m_listener.onTrackTimeSkip(m_mediaType, startTime);
        } else {
            m_log.warn("waiting for keyframe");
            return false;
        }
    }

    if (format) {
        m_format = format;
    } else {
        m_log.warn("reconfigured without new media format");
    }

    bool discontinuous = m_seeking;
    MediaTime renderedTime = renderer.getRenderedTime();
    if (!m_seeking && renderedTime.valid()) {
        if ((renderedTime - sample->presentationTime).absolute() > MediaTime(1.0)) {
            m_log.warn("sample time discontinuity detected input %lld us, current %lld us",
                sample->presentationTime.microseconds().count(), renderedTime.microseconds().count());
            discontinuous = true;
            m_listener.onTrackTimeDiscontinuity(m_mediaType, sample->presentationTime);
        }
    }

    renderer.configure(m_format, discontinuous);
    // if configure fails m_paused will be set to true
    if (m_paused) {
        return false;
    }
    m_listener.onTrackConfigured(m_format);
    return true;
}

void TrackSink::dequeueActions(TrackRenderer& renderer)
{
    std::lock_guard<Mutex> lock(m_mutex);
    while (!m_preConfigureActions.empty()) {
        auto action = m_preConfigureActions.front();
        m_preConfigureActions.pop();
        action(renderer);
    }
    if (renderer.isConfigured()) {
        while (!m_postConfigureActions.empty()) {
            auto action = m_postConfigureActions.front();
            m_postConfigureActions.pop();
            action(renderer);
        }
    }
}

void TrackSink::prepare()
{
    if (m_isVideo) {
        std::lock_guard<Mutex> lock(m_mutex);
        if (m_seeking) {
            m_buffer.seek(m_seekTime);
            // on the video track attempt frame accurate seek to the given time
            // seek time maybe zero but it may not be present in the sample buffer
            if (m_seekTime == MediaTime::zero() || m_buffer.hasPresentationTime(m_seekTime)) {
                m_paused = false;
                m_queueCondition.notify_one();
                return;
            }
        }
        // if not found set as prepared
    }
    m_listener.onTrackPrepared(m_mediaType);
}

void TrackSink::onSeekCompleted()
{
    {
        std::lock_guard<Mutex> lock(m_mutex);
        m_seekTime = MediaTime::zero();
        m_seeking = false;
        m_paused = true;
        if (!m_buffer.isEmpty() && m_buffer.front().sample) {
            m_log.info("synced at %lld", m_buffer.front().sample->decodeTime.microseconds().count());
        }
    }
    m_listener.onTrackPrepared(m_mediaType);
}

void TrackSink::updateIdleState(TrackRenderer& renderer, bool idle)
{
    if (m_idle != idle) {
        {
            std::lock_guard<Mutex> lock(m_mutex);
            m_idle = idle;
        }

        renderer.updateState(idle, m_paused);

        if (m_idle) {
            m_idleCondition.notify_all();
            m_listener.onTrackIdle(m_mediaType);
        }
    }
}

void TrackSink::notifyError(ErrorSource source, MediaResult result, const std::string& message)
{
    if (!m_run.isCancelled() && result != MediaResult::Ok) {
        m_listener.onTrackError(m_mediaType, Error(source, result, m_mediaType.type + " : " + message));

        m_paused = true; // don't continue to process samples (player may flush or play)
        m_waitCondition.notify_one(); // unblock any waits
    }
}

void TrackSink::configure(const std::shared_ptr<const MediaFormat>& format)
{
    if (format) {
        std::lock_guard<Mutex> lock(m_mutex);
        m_buffer.push(format);
    }
}

void TrackSink::enqueue(const std::shared_ptr<MediaSampleBuffer>& sample)
{
    if (sample) {
        std::lock_guard<Mutex> lock(m_mutex);
        m_buffer.push(sample);

        if (m_idle && !m_paused) {
            m_queueCondition.notify_one();
        }
    }
}

void TrackSink::play()
{
    {
        std::lock_guard<Mutex> lock(m_mutex);
        if (m_paused) {
            m_paused = false;
            if (m_seeking) {
                m_buffer.seek(m_seekTime);
            }
        }
    }

    m_queueCondition.notify_one();
}

void TrackSink::pause()
{
    std::unique_lock<Mutex> lock(m_mutex);
    awaitIdle(lock);
}

void TrackSink::flush()
{
    queueRendererAction([](TrackRenderer& renderer) {
        renderer.flush();
    },
        false);
    std::unique_lock<Mutex> lock(m_mutex);
    m_buffer.clear();
    awaitIdle(lock);
}

void TrackSink::remove(const TimeRange& range)
{
    std::lock_guard<Mutex> lock(m_mutex);
    m_buffer.remove(range);
}

void TrackSink::seekTo(MediaTime time)
{
    {
        std::lock_guard<Mutex> lock(m_mutex);
        m_seekTime = time;
        m_seeking = true;
        m_buffer.seek(m_seekTime);
    }

    m_waitCondition.notify_one(); // unblocks any frame syncing
}

bool TrackSink::isIdle()
{
    std::lock_guard<Mutex> lock(m_mutex);
    return m_idle;
}

void TrackSink::setPlaybackRate(float rate)
{
    queueRendererAction([rate](TrackRenderer& renderer) {
        renderer.setPlaybackRate(rate);
    },
        false);
}

void TrackSink::setSurface(void* surface)
{
    queueRendererAction([this, surface](TrackRenderer& renderer) {
        renderer.setSurface(surface);
        m_surface = surface;
        m_surfaceChanged.notify_one();
    },
        true);
    // not this method be synchronous to the ui code to ensure the surface is erased or switched
    // before the end of the function call
    std::unique_lock<Mutex> lock(m_mutex);
    m_surfaceChanged.wait(lock, [this, surface]() { return m_idle || m_run.isCancelled() || m_surface == surface; });
}

void TrackSink::setVolume(float volume)
{
    queueRendererAction([volume](TrackRenderer& renderer) {
        renderer.setVolume(volume);
    },
        false);
}

void TrackSink::awaitIdle(std::unique_lock<Mutex>& lock)
{
    m_paused = true;
    m_waitCondition.notify_one(); // unblocks any frame syncing
    // wait for process queue to idle
    m_idleCondition.wait(lock, [this]() { return m_idle || m_run.isCancelled(); });
}

void TrackSink::queueRendererAction(const std::function<void(TrackRenderer&)>& action, bool preconfigure)
{
    if (action) {
        std::lock_guard<Mutex> lock(m_mutex);
        if (preconfigure) {
            m_preConfigureActions.push(action);
        } else {
            m_postConfigureActions.push(action);
        }
    }
}

void TrackSink::onDecodeError(MediaResult result, const std::string& message)
{
    notifyError(ErrorSource::Decode, result, message);
}

void TrackSink::onRenderError(MediaResult result, const std::string& message)
{
    notifyError(ErrorSource::Render, result, message);
}

void TrackSink::onRenderTimeUpdate(MediaTime time)
{
    if (time.valid()) {
        m_listener.onTrackTimeUpdate(m_mediaType, time);
    }
}

void TrackSink::onMetadataSample(std::shared_ptr<const MediaSampleBuffer> sample)
{
    m_listener.onTrackMetadataSample(sample);
}

void TrackSink::onStatisticsUpdated(const Statistics& statistics)
{
    m_listener.onTrackStatistics(m_mediaType, statistics);
}

bool TrackSink::onTimedWait(MediaTime time)
{
    if (time > MediaTime::zero()) {
        std::unique_lock<Mutex> lock(m_mutex);
        return !m_waitCondition.wait_for(lock, time.microseconds(), [this]() { return m_run.isCancelled() || m_paused; });
    }

    return true;
}
}
