#pragma once

#include "CancellationFlag.hpp"
#include "MediaClock.hpp"
#include "TrackBuffer.hpp"
#include "TrackRenderer.hpp"
#include "media/SourceFormat.hpp"
#include "playercore/platform/NativePlatform.hpp"
#include "util/Concurrency.hpp"
#include <atomic>
#include <queue>
#include <string>

namespace twitch {
class TrackSink : private TrackRenderer::Listener {
public:
    class Listener {
    public:
        virtual ~Listener() = default;
        virtual void onTrackError(const MediaType& type, const Error& error) = 0;
        virtual void onTrackConfigured(std::shared_ptr<const MediaFormat> format) = 0;
        virtual void onTrackMetadataSample(std::shared_ptr<const MediaSampleBuffer> sample) = 0;
        virtual void onTrackPrepared(const MediaType& type) = 0;
        virtual void onTrackRecoverableError(const Error& error) = 0;
        virtual void onTrackTimeDiscontinuity(const MediaType& type, MediaTime time) = 0;
        virtual void onTrackTimeSkip(const MediaType& type, MediaTime time) = 0;
        virtual void onTrackTimeUpdate(const MediaType& type, MediaTime time) = 0;
        virtual void onTrackIdle(const MediaType& type) = 0;
        virtual void onTrackStatistics(const MediaType& type, const Statistics& statistics) = 0;
    };

    TrackSink(NativePlatform& platform, Listener& listener, MediaClock& clock, const std::shared_ptr<MediaFormat>& format);
    ~TrackSink() override;
    TrackSink(const TrackSink&) = delete;
    const TrackSink& operator=(const TrackSink&) = delete;

    void configure(const std::shared_ptr<const MediaFormat>& format);
    void enqueue(const std::shared_ptr<MediaSampleBuffer>& sample);
    void prepare();
    void play();
    void pause();
    void flush();
    void remove(const TimeRange& range);
    void seekTo(MediaTime time);
    bool isIdle();
    void setPlaybackRate(float rate);
    void setSurface(void* surface);
    void setVolume(float volume);
    bool isEmpty() { return m_buffer.isEmpty(); }

private:
    void onDecodeError(MediaResult result, const std::string& message) override;
    void onRenderError(MediaResult result, const std::string& message) override;
    void onRenderTimeUpdate(MediaTime time) override;
    void onMetadataSample(std::shared_ptr<const MediaSampleBuffer> sample) override;
    void onStatisticsUpdated(const Statistics& statistics) override;
    bool onTimedWait(MediaTime time) override;

    bool rendererConfigure(TrackRenderer& renderer, const std::shared_ptr<const MediaFormat>& format, const std::shared_ptr<const MediaSampleBuffer>& sample);
    void onSeekCompleted();
    void dequeueActions(TrackRenderer& renderer);
    void updateIdleState(TrackRenderer& renderer, bool idle);
    void processQueue();
    void notifyError(ErrorSource source, MediaResult result, const std::string& message);
    void awaitIdle(std::unique_lock<Mutex>& lock);
    void queueRendererAction(const std::function<void(TrackRenderer&)>& action, bool preconfigure);

    NativePlatform& m_platform;
    PrefixedLog m_log;
    Listener& m_listener;
    CancellationFlag m_run;
    MediaClock& m_clock;
    std::shared_ptr<Scheduler> m_scheduler;
    std::shared_ptr<const MediaFormat> m_format;
    MediaType m_mediaType;
    TrackBuffer m_buffer;
    std::queue<std::function<void(TrackRenderer&)>> m_postConfigureActions;
    std::queue<std::function<void(TrackRenderer&)>> m_preConfigureActions;
    MediaTime m_seekTime;
    bool m_paused;
    bool m_idle;
    bool m_seeking;
    Mutex m_mutex;
    ConditionVariable m_queueCondition;
    ConditionVariable m_waitCondition;
    ConditionVariable m_idleCondition;
    ConditionVariable m_surfaceChanged;
    bool m_isVideo;
    void* m_surface;
};
}
