#include "ChannelSource.hpp"
#include "media/ProtectionSystem.hpp"
#include "media/SourceFormat.hpp"
#include "player/PassthroughSource.hpp"
#include "sink/DrmProvider.hpp"
#include "util/UriBuilder.hpp"

namespace twitch {
ChannelSource::ChannelSource(MediaSource::Listener& listener,
    std::shared_ptr<Platform> platform,
    std::shared_ptr<Scheduler> scheduler,
    std::shared_ptr<TokenHandler> tokenHandler,
    const std::string& url,
    hls::HlsSource::Options options,
    bool remote)
    : m_listener(listener)
    , m_platform(platform)
    , m_scheduler(scheduler)
    , m_httpClient(platform->createAsyncHttpClient(scheduler))
    , m_tokenHandler(std::move(tokenHandler))
    , m_log(platform->getLog())
    , m_url(url)
    , m_link(url)
    , m_options(options)
    , m_tokenRequest("AccessToken")
    , m_playlistRequest("Playlist")
    , m_serverAdRequest("ServerAd")
    , m_remote(remote)
{
    if (m_link.getId().empty()) {
        if (m_platform->getName() == "web") {
            // hack to set the cdm parameter in the usher master playlist request
            if (m_url.find("usher") != std::string::npos && m_url.find("cdm=") == std::string::npos) {
                std::string cdm = getCDMParameter();
                if (!cdm.empty()) {
                    m_url += "&cdm=" + cdm;
                }
            }
        }

        // LVS streams will have prefetch_segments=true parameter
        if (m_url.find("api/lvs/hls/lvs") != std::string::npos && m_url.find("prefetch_segments") == std::string::npos) {
            std::string paramSeparator = m_url.find(".m3u8?") != std::string::npos ? "&" : "?";
            m_url += paramSeparator + "prefetch_segments=true";
        }
    }
}

ChannelSource::~ChannelSource()
{
    close();
}

void ChannelSource::open()
{
    if (m_source) {
        m_source->open();
        return;
    }

    if (m_link.getId().empty()) {
        createSource(m_url, true);
        return;
    }

    // if a url resolver is set use that to get the master url, otherwise the url is the master url
    auto& cachedToken = m_tokenHandler->getToken(m_link);
    if (cachedToken.sig.empty()) {
        std::string url = m_tokenHandler->createAccessTokenRequest(m_link);
        // m_log->info("access token request %s to %s", m_url.c_str(), url.c_str());
        m_tokenRequest.setUrl(url);
        sendRequest(m_tokenRequest, [this](const std::string& response) {
            onAccessTokenResponse(response);
        });
        // defer until look up is done
        return;
    } else {
        createSource(createMasterPlaylistUrl(cachedToken.token, cachedToken.sig), true);
    }
}

void ChannelSource::close()
{
    m_tokenRequest.cancel();
    m_serverAdRequest.cancel();
    if (m_source) {
        m_source->close();
    }
}

void ChannelSource::seekTo(MediaTime time)
{
    if (m_source) {
        return m_source->seekTo(time);
    }
}

void ChannelSource::read(const TimeRange& range)
{
    if (m_source) {
        return m_source->read(range);
    }
}

bool ChannelSource::isLive() const
{
    return m_source && m_source->isLive();
}

bool ChannelSource::isPassthrough() const
{
    return m_source && m_source->isPassthrough();
}

bool ChannelSource::isSeekable() const
{
    return m_source && m_source->isSeekable();
}

void ChannelSource::setQuality(const Quality& quality, bool adaptive)
{
    if (m_source) {
        m_source->setQuality(quality, adaptive);
    }
}

const Quality& ChannelSource::getQuality() const
{
    static Quality empty;
    if (m_source) {
        return m_source->getQuality();
    } else {
        return empty;
    }
}

const std::vector<Quality>& ChannelSource::getQualities() const
{
    static std::vector<Quality> empty;
    if (m_source) {
        return m_source->getQualities();
    } else {
        return empty;
    }
}

void ChannelSource::setReadTimeout(MediaTime time)
{
    if (m_source) {
        m_source->setReadTimeout(time);
    }
}

void ChannelSource::setLowLatencyEnabled(bool enable)
{
    if (m_source) {
        m_source->setLowLatencyEnabled(enable);
    }
}

MediaTime ChannelSource::getDuration() const
{
    return m_source ? m_source->getDuration() : MediaTime::zero();
}

std::string ChannelSource::createMasterPlaylistUrl(const std::string& token, const std::string& sig)
{
    UriBuilder builder("https", "usher.ttvnw.net");
    //UriBuilder builder("http", "usher.staging.justin.tv");
    std::string path;

    switch (m_link.getContentType()) {
    case TwitchLink::ContentType ::Live:
        builder.setEncodedParameter("token", token);
        builder.setParameter("sig", sig);
        builder.setParameter("fast_bread", "true"); // enables prefetch segments
        path = "api/channel/hls/";
        break;
    case TwitchLink::ContentType::VOD:
        builder.setEncodedParameter("nauth", token);
        builder.setParameter("nauthsig", sig);
        path = "vod/";
        break;
    case TwitchLink::ContentType::Clip:
    case TwitchLink::ContentType::Unknown:
        break;
    }

    path += m_link.getId();
    path += ".m3u8";
    builder.setPath(path);
    builder.setParameter("allow_audio_only", "true");
    builder.setParameter("allow_source", "true");
    builder.setParameter("player_backend", "mediaplayer");
    builder.setParameter("player_type", m_tokenHandler->getPlayerType());
    builder.setParameter("playlist_include_framerate", "true");
    builder.setParameter("reassignments_supported", "true");

    // add supported extra codecs
    if (m_tokenHandler->isPlayerCoreClientId()) {
        std::string codecs = "avc1";
        if (m_platform->getSupportedMediaTypes().count(MediaType::Video_VP9)) {
            codecs = "vp09," + codecs;
        }
        builder.setParameter("supported_codecs", codecs);
    }
    // internal testing flags (internal network only)
    //builder.setParameter("force_weaver", "true");
    //builder.setParameter("force_manifest_node", "video-weaver-staging.sjc02");
    //builder.setParameter("force_vizima", "true");

    // forces ads
    //builder.setParameter("force_preroll", "true"); // pre-rolls
    //builder.setParameter("force_midroll", "true"); // mid-rolls

    std::string cdm = getCDMParameter();
    if (!cdm.empty()) {
        builder.setParameter("cdm", cdm);
    }

    for (const auto& param : m_link.getQueryParameters()) {
        builder.setParameter(param.first, param.second);
    }
    return builder.build();
}

std::string ChannelSource::getCDMParameter()
{
    std::string cdm;
    for (const auto& systemId : m_platform->getSupportedProtectionSystems()) {
        Uuid uuid = Uuid::fromBytes(systemId);
        if (uuid == media::ProtectionSystemWidevine) {
            cdm = "wv";
            break;
        } else if (uuid == media::ProtectionSystemPlayReady) {
            cdm = "pr";
            break;
        } else if (uuid == media::ProtectionSystemFairPlay) {
            cdm = "fp";
            break;
        }
    }
    return cdm;
}

void ChannelSource::createSource(const std::string& url, bool open)
{
    // first check the platform, since it can create a custom source (e.g. passthrough)
    if (m_remote) {
        m_source.reset(new PassthroughSource(m_listener, MediaType::Application_MPEG_URL, url, true));
    } else {
        m_source = m_platform->createSource(url, MediaType::Application_MPEG_URL, m_listener, m_scheduler);
    }

    if (!m_source) {
        m_source.reset(new hls::HlsSource(m_listener, m_platform, m_scheduler, m_httpClient, url, m_options));
    }

    if (open && m_source) {
        m_source->open();
    }
}

void ChannelSource::onRequestError(MediaRequest& request, std::function<void(const std::string& content)> handler, int error)
{
    request.onNetworkError(error);
    Error ioError(ErrorSource::Playlist, MediaResult(MediaResult::ErrorNetwork, error), "Access token error");

    if (!request.isRetryable()) {
        m_listener.onSourceError(ioError);
    } else {
        m_listener.onSourceRecoverableError(ioError);
        request.retry(*m_scheduler, [this, &request, handler]() { sendRequest(request, handler); });
    }
}

void ChannelSource::onAccessTokenResponse(const std::string& content)
{
    auto& token = m_tokenHandler->parseTokenResponse(m_link, content);
    std::string url = createMasterPlaylistUrl(token.token, token.sig);

    createSource(url, false);

    if (m_source->isPassthrough()) {
        m_playlistRequest.setUrl(url);

        sendRequest(m_playlistRequest, [this, url](const std::string& response) {
            onMasterPlaylist(response);
            m_source->open();
        });
    } else {
        m_source->open();
    }
}

void ChannelSource::onMasterPlaylist(const std::string& content)
{
    // passthrough mode download the master playlist ourselves
    if (m_master.parse(content, m_playlistRequest.getUrl())) {
        m_listener.onSourceSessionData(m_master.getSessionData());
    } else {
        m_listener.onSourceError(
            Error(ErrorSource::Playlist, MediaResult::ErrorInvalidData, "Failed to read master playlist"));
    }
}

void ChannelSource::sendRequest(MediaRequest& request, std::function<void(const std::string& content)> handler)
{
    auto httpRequest = m_httpClient->createRequest(request.getUrl(), HttpMethod::GET);

    for (const auto& entry : m_tokenHandler->getHeaders()) {
        httpRequest->setHeader(entry.first, entry.second);
    }

    if (request.getType() == "Playlist") {
        std::string accept;
        for (const auto& media : {
                 MediaType::Application_MPEG_URL,
                 MediaType::Application_Apple_MPEG_URL,
                 MediaType::Application_Json,
                 MediaType::Text_Plain }) {
            if (!accept.empty()) {
                accept += ", ";
            }
            accept += media.name;
        }
        httpRequest->setHeader("Accept", accept);
    }

    auto networkError = [this, &request, handler](int error) {
        onRequestError(request, handler, error);
    };

    request.onRequest(httpRequest);
    m_httpClient->send(
        httpRequest, [this, &request, handler, networkError](const std::shared_ptr<HttpResponse>& response) {
            request.onResponse(*response);

            if (response->isSuccess()) {
                request.readString(
                    *response, [&request, handler](const std::string& content) {
                        request.onCompleted();
                        handler(content);
                    },
                    networkError);
            } else {
                int status = response->getStatus();
                if (status == HttpStatusForbidden || status == HttpStatusGone) {
                    m_tokenHandler->removeToken(m_link);
                    std::string url = m_tokenHandler->createAccessTokenRequest(m_link);
                    m_tokenRequest.setUrl(url);
                    networkError(status); // retry (if attempts remaining)
                } else {
                    m_listener.onSourceError(
                        Error(ErrorSource::Playlist, MediaResult(MediaResult::ErrorNetwork, status),
                            "Failed to get url"));
                }
            }
        },
        networkError);
}

void ChannelSource::requestServerAd(const std::string& url)
{
    m_serverAdRequest.setUrl(url);
    sendRequest(m_serverAdRequest, [](const std::string& response) {
        (void)response;
    });
}
}
