#pragma once

#include "TokenHandler.hpp"
#include "hls/HlsSource.hpp"
#include "hls/MasterPlaylist.hpp"
#include "player/MediaRequest.hpp"
#include "playercore/platform/Platform.hpp"

namespace twitch {
/**
 * Source that takes a url input parameter, matches the url against the twitch link format, e.g.
 * http://twitch.tv/<channelname>
 * http://twitch.tv/<channelname>/v/<vodId> and http://twitch.tv/videos/<vodId>
 *
 * If the url matches then it's parsed and an nauth request is made to get the master playlist url.
 * On success an HlsSource instance is created to play the retrieved playlist url and this class
 * acts as a wrapper around that HlsSource instance.
 *
 * If the input url doesn't match the link format it's assumed to be an HLS playlist url and a
 * HlsSource instance is created immediately.
 */
class ChannelSource : public MediaSource {
public:
    ChannelSource(MediaSource::Listener& listener,
        std::shared_ptr<Platform> platform,
        std::shared_ptr<Scheduler> scheduler,
        std::shared_ptr<TokenHandler> tokenHandler,
        const std::string& url,
        hls::HlsSource::Options options,
        bool remote);
    ~ChannelSource() override;
    ChannelSource(const ChannelSource&) = delete;
    const ChannelSource& operator=(const ChannelSource&) = delete;

    void open() override;
    void close() override;
    void seekTo(MediaTime time) override;
    void read(const TimeRange& range) override;
    bool isLive() const override;
    bool isPassthrough() const override;
    bool isSeekable() const override;
    MediaTime getDuration() const override;
    const Quality& getQuality() const override;
    void setQuality(const Quality& quality, bool adaptive) override;
    const std::vector<Quality>& getQualities() const override;
    void setReadTimeout(MediaTime time) override;
    void setLowLatencyEnabled(bool enable) override;
    void requestServerAd(const std::string& url);

private:
    void createSource(const std::string& url, bool open);
    std::string createMasterPlaylistUrl(const std::string& token, const std::string& sig);
    std::string getCDMParameter();
    void onRequestError(MediaRequest& request, std::function<void(const std::string& content)> handler, int error);
    void onAccessTokenResponse(const std::string& content);
    void onMasterPlaylist(const std::string& content);
    void sendRequest(MediaRequest& request, std::function<void(const std::string& playlist)> handler);

    MediaSource::Listener& m_listener;
    std::shared_ptr<Platform> m_platform;
    std::shared_ptr<Scheduler> m_scheduler;
    std::shared_ptr<HttpClient> m_httpClient;
    std::unique_ptr<MediaSource> m_source;
    std::shared_ptr<TokenHandler> m_tokenHandler;
    std::shared_ptr<Log> m_log;
    std::string m_url;
    TwitchLink m_link;
    const hls::HlsSource::Options m_options;
    MediaRequest m_tokenRequest;
    MediaRequest m_playlistRequest;
    MediaRequest m_serverAdRequest;
    hls::MasterPlaylist m_master;
    std::vector<Quality> m_qualities;
    bool m_remote;
};
}
