#include "ClipSource.hpp"
#include "util/UriBuilder.hpp"
#include <json11.hpp>

namespace twitch {
ClipSource::ClipSource(MediaSource::Listener& listener,
    std::shared_ptr<Platform> platform,
    std::shared_ptr<Scheduler> scheduler,
    const TwitchLink& link)
    : m_listener(listener)
    , m_platform(platform)
    , m_scheduler(scheduler)
    , m_httpClient(platform->createAsyncHttpClient(scheduler))
    , m_link(link)
    , m_statusRequest("ClipStatus")
{
}

ClipSource::~ClipSource()
{
    close();
}

void ClipSource::open()
{
    if (m_link.getContentType() != TwitchLink::ContentType::Clip) {
        m_listener.onSourceError(
            Error(ErrorSource::Source, MediaResult::ErrorInvalidData, "Invalid clip link"));
        return;
    }
    if (!m_statusRequest.isPending() || !m_statusRequest.isComplete()) {
        // get the clip metadata
        UriBuilder builder("https", "clips.twitch.tv");
        std::string path = "api/v2/clips/";
        path += m_link.getId();
        path += "/status";
        builder.setPath(path);
        m_statusRequest.setUrl(builder.build());
        sendStatusRequest();
    }
}

void ClipSource::close()
{
    m_statusRequest.cancel();
    if (m_source) {
        m_source->close();
    }
}

void ClipSource::seekTo(MediaTime time)
{
    if (m_source) {
        return m_source->seekTo(time);
    }
}

void ClipSource::read(const TimeRange& range)
{
    if (m_source) {
        return m_source->read(range);
    }
}

bool ClipSource::isPassthrough() const
{
    return m_source && m_source->isPassthrough();
}

bool ClipSource::isSeekable() const
{
    return m_source && m_source->isSeekable();
}

void ClipSource::setQuality(const Quality& quality, bool adaptive)
{
    (void)adaptive;
    if (quality != m_selected && m_sources.count(quality.name)) {
        m_selected = quality;
        createSource(m_sources[quality.name]);
    }
}

const Quality& ClipSource::getQuality() const
{
    return m_selected;
}

const std::vector<Quality>& ClipSource::getQualities() const
{
    return m_qualities;
}

void ClipSource::setReadTimeout(MediaTime time)
{
    if (m_source) {
        m_source->setReadTimeout(time);
    }
}

MediaTime ClipSource::getDuration() const
{
    return m_source ? m_source->getDuration() : MediaTime::zero();
}

void ClipSource::createSource(const std::string& url)
{
    m_source = m_platform->createSource(url, MediaType::Video_MP4, m_listener, m_scheduler);
    if (m_source) {
        m_source->open();
    }
}

void ClipSource::onClipStatus(const std::string& content)
{
    std::string err;
    json11::Json json = json11::Json::parse(content, err);
    const std::string& status = json["status"].string_value();

    if (status != "created") {
        m_listener.onSourceError(Error(ErrorSource::Source,
            MediaResult::ErrorNotAvailable, "Clip not created"));
        return;
    }

    // populate quality options and source urls
    const json11::Json::array& options = json["quality_options"].array_items();
    for (auto& option : options) {
        auto& name = option["quality"].string_value();
        auto& source = option["source"].string_value();
        auto framerate = option["frame_rate"].int_value();
        Quality quality {};
        quality.name = name;
        quality.framerate = static_cast<float>(framerate);
        quality.autoSelect = false; // make sure abr is disabled
        // map the name to source url
        m_sources[name] = source;
        m_qualities.push_back(quality);
    }

    m_listener.onSourceOpened();
}

void ClipSource::sendStatusRequest()
{
    auto httpRequest = m_httpClient->createRequest(m_statusRequest.getUrl(), HttpMethod::GET);
    m_statusRequest.onRequest(httpRequest);
    m_httpClient->send(
        httpRequest, [this](std::shared_ptr<HttpResponse> response) {
        m_statusRequest.onResponse(*response);
        if (response->isSuccess()) {
            m_statusRequest.readString(*response, [this](const std::string& content) {
                onClipStatus(content);
            },
           [this](int error) {
               onRequestError(error);
           });
        } else {
            m_listener.onSourceError(Error(ErrorSource::Source,
                MediaResult(MediaResult::ErrorNotAvailable,
                            response->getStatus()), "Clip not available"));
        } }, [this](int error) { onRequestError(error); });
}

void ClipSource::onRequestError(int error)
{
    m_statusRequest.onNetworkError(error);
    Error ioError(ErrorSource::Source, MediaResult(MediaResult::ErrorNetwork, error), "network error");

    if (!m_statusRequest.isRetryable()) {
        m_listener.onSourceError(ioError);
    } else {
        m_listener.onSourceRecoverableError(ioError);
        m_statusRequest.retry(*m_scheduler, [this]() { sendStatusRequest(); });
    }
}
}
