#include "DrmKeyOS.hpp"
#include "debug/trace.hpp"
#include "media/ProtectionSystem.hpp"
#include "util/UriBuilder.hpp"

namespace twitch {
DrmKeyOs::DrmKeyOs(DrmProvider::Listener& listener, std::shared_ptr<HttpClient> httpClient, const Uuid& systemId)
    : m_listener(listener)
    , m_httpClient(std::move(httpClient))
    , m_systemId(systemId)
    , m_request("AuthXML")
{
}

std::string DrmKeyOs::getLicenseAcquisitionUrl()
{
    if (m_systemId == media::ProtectionSystemFairPlay) {
        return "https://fp-keyos-twitch.licensekeyserver.com/getkey";
    } else if (m_systemId == media::ProtectionSystemPlayReady) {
        return "https://pr-keyos-twitch.licensekeyserver.com/core/rightsmanager.asmx";
    } else if (m_systemId == media::ProtectionSystemWidevine) {
        return "https://wv-keyos-twitch.licensekeyserver.com";
    }
    return "";
}

void DrmKeyOs::onLicenseRequest(HttpRequest& request)
{
    if (!m_customdata.empty()) {
        request.setHeader("customdata", m_customdata);
    }
}

void DrmKeyOs::prepare(const MediaFormat& format)
{
    std::string path = format.getPath();

    if (m_contentPath != path) {
        m_contentPath = path;
        std::string channel;
        // extract the channel name from the usher url
        auto start = m_contentPath.find_last_of('/');
        if (start != std::string::npos) {
            auto end = m_contentPath.size() - 1;
            auto index = m_contentPath.find(".m3u8");
            if (index != std::string::npos) {
                end = index;
            }
            channel = m_contentPath.substr(start + 1, end - start - 1);
        }

        if (!channel.empty()) {
            std::map<std::string, std::string> parameters;
            UriBuilder::getParametersFromUrl(m_contentPath, parameters);
            const std::string& token = parameters["token"];
            const std::string& sig = parameters["sig"];
            requestAuthXML(channel, token, sig);
        } else {
            Error error(ErrorSource::Source, MediaResult::ErrorInvalidData, "Failed to get twitch channel from path");
            m_listener.onProviderError(error);
        }
    } else {
        m_listener.onProviderPrepared();
    }
}

void DrmKeyOs::requestAuthXML(const std::string& channel, const std::string& token, const std::string& sig)
{
    UriBuilder builder("https", "vizima.twitch.tv");
    builder.setPath("api/authxml/" + channel);
    builder.setParameter("token", token);
    builder.setParameter("sig", sig);

    std::string url = builder.build();
    auto httpRequest = m_httpClient->createRequest(url, HttpMethod::GET);

    m_request.onRequest(httpRequest);
    m_request.setUrl(url);
    m_httpClient->send(
        httpRequest,
        [=](const std::shared_ptr<HttpResponse>& response) {
            // handle http response
            onResponse(*response);
        },
        [=](int error) {
            // handle request error
            onRequestError(MediaResult::ErrorNetworkIO, error);
        });
}

void DrmKeyOs::onResponse(HttpResponse& response)
{
    m_request.onResponse(response);

    if (response.isSuccess()) {
        std::shared_ptr<std::vector<uint8_t>> bytes = std::make_shared<std::vector<uint8_t>>();
        response.read([this, bytes](const uint8_t* data, size_t size, bool endOfStream) mutable {
              bytes->insert(bytes->end(), data, data + size);
              if (endOfStream) {
                  m_customdata = std::string(bytes->begin(), bytes->end());
                  m_request.onCompleted();
                  // ready to make key request
                  m_listener.onProviderPrepared();
              } },
            [=](int code) {
                onRequestError(MediaResult::ErrorNetworkIO, code);
            });
    } else {
        onRequestError(MediaResult::ErrorNetwork, response.getStatus());
    }
}

void DrmKeyOs::onRequestError(MediaResult result, int error)
{
    Error authError(ErrorSource::Source, MediaResult(result, error), "Failed to get AuthXML");
    m_listener.onProviderError(authError);
}
}
