#include "MetadataParser.hpp"
#include "debug/trace.hpp"
#include <cstdlib>

namespace twitch {
const std::string MetadataParser::StitchedAdClass("twitch-stitched-ad");
const std::string MetadataParser::AssignmentClass("twitch-assignment");
const std::string MetadataParser::StreamSourceClass("twitch-stream-source");
const std::string MetadataParser::AdTrigger("twitch-trigger");

void MetadataParser::onSample(const MediaSampleBuffer& sample, MetadataParser::Listener& listener)
{
    std::string contents(sample.buffer.data(), sample.buffer.data() + sample.buffer.size());
    using namespace json11;
    std::string err;
    json11::Json json = json11::Json::parse(contents, err);
    std::string metadataClass = json["CLASS"].string_value();

    if (!json["ID3"].array_items().empty()) {
        Json::array array = json["ID3"].array_items();
        for (const auto& item : array) {
            if (item["id"].string_value() == "TXXX" && item["desc"] == "segmentmetadata") {
                Json::array info = item["info"].array_items();
                for (auto& inner : info) {
                    Json txxx = Json::parse(inner.string_value(), err);
                    if (err.empty()) {
                        std::string cmd = txxx["cmd"].string_value();
                        // indicates latency data
                        if (cmd == "ld_lat_data") {
                            double streamOffset = txxx["stream_offset"].number_value();
                            // note these values are more accurate than the TDTG/TDEN
                            uint64_t transcodeSend = static_cast<uint64_t>(txxx["transc_s"].number_value());
                            uint64_t transcodeReceive = static_cast<uint64_t>(txxx["transc_r"].number_value());
                            listener.onMetaLatencyTiming(MediaTime(streamOffset),
                                MediaTime(transcodeReceive, std::milli::den),
                                MediaTime(transcodeSend, std::milli::den));
                        }
                        // indicates commercial mid-roll
                        else if (cmd == "commercial") {
                            int length = std::stoi(txxx["length"].string_value());
                            if (length > 0) {
                                listener.onMetaAdBreakRequested(MediaTime(length, 1));
                            }
                        }
                    } else {
                        TRACE_WARN("Invalid segmentmetadata json: %s", err.c_str());
                    }
                }
            }
        }
    } else if (metadataClass == AssignmentClass || metadataClass == AdTrigger) {
        // edge reassignment update session info for reporting
        const std::string AttributePrefix("X-TV-TWITCH-");
        std::map<std::string, std::string> values;
        for (const auto& entry : json.object_items()) {
            if (entry.first.compare(0, AttributePrefix.length(), AttributePrefix) == 0) {
                values[entry.first] = entry.second.string_value();
            }
        }
        listener.onMetaSessionData(values);
    } else if (metadataClass == StitchedAdClass) {
        std::string rollType = json["X-TV-TWITCH-AD-ROLL-TYPE"].string_value();
        AdRollType type = AdRollType::Pre;
        if (rollType == "PREROLL") {
            type = AdRollType::Pre;
        } else if (rollType == "MIDROLL") {
            type = AdRollType::Mid;
        } else if (rollType == "POSTROLL") {
            type = AdRollType::Post;
        }
        m_adRollType = type;
        listener.onMetaServerAdBreakStart(type);
    } else if (metadataClass == StreamSourceClass) {
        if ((json["X-TV-TWITCH-STREAM-SOURCE"].string_value() == "live")) {
            listener.onMetaServerAdBreakEnd();
        }
    }
}
}
