#include "SessionData.hpp"
#include <chrono>

namespace twitch {
void SessionData::clear()
{
    m_info.clear();
}

void SessionData::reset(const std::map<std::string, std::string>& properties)
{
    m_info = properties;
}

void SessionData::update(const std::map<std::string, std::string>& properties)
{
    // when updating the info from the HLS media playlist the attributes will be prefixed
    const std::string AttributePrefix("X-TV-TWITCH-");

    for (const auto& entry : properties) {
        if (entry.first.compare(0, AttributePrefix.length(), AttributePrefix) == 0) {
            // strip the key prefix and overwrite the value
            std::string key = entry.first.substr(AttributePrefix.length());
            m_info[key] = entry.second;
        }
    }
}

bool SessionData::isLowLatency() const
{
    // checks if the stream has prefetch segment support
    auto it = m_info.find("FUTURE");
    return it != m_info.end() && "true" == it->second;
}

bool SessionData::isUltraLowLatency() const
{
    // for syncd low latency transcodes for Amazon Trivia and 2s sync demo channels
    auto entry = m_info.find("TRANSCODESTACK");
    if (entry != m_info.end()) {
        const std::string prefix("2018TranscodePinot");
        return entry->second.compare(0, prefix.size(), prefix) == 0;
    }
    return false;
}

bool SessionData::isHoldbackGroup() const
{
    // Consider in holdback group if key isn't present
    auto it = m_info.find("B");
    return it == m_info.end() || it->second == "true";
}

double SessionData::getServerTime() const
{
    auto entry = m_info.find("SERVER-TIME");
    if (entry != m_info.end()) {
        return std::stod(entry->second); // value is unix time (floating point)
    }

    return 0.0;
}

std::string SessionData::getAdTriggerUrl() const
{
    // key is X-TV-TWITCH-TRIGGER-URL prefix is stripped in update()
    auto entry = m_info.find("TRIGGER-URL");
    if (entry != m_info.end()) {
        return entry->second;
    }
    return "";
}
}
