#include "TokenHandler.hpp"
#include "util/UriBuilder.hpp"
#include "util/Uuid.hpp"
#include <algorithm>
#include <chrono>

namespace twitch {
const std::string PlayerCoreClientId = "7xowrgquc8495rq4w5p79aeiwzxglu8";
const std::string AcceptJsonVersion = "application/vnd.twitchtv.v5+json";

TokenHandler::TokenHandler(std::shared_ptr<Platform> platform)
    : m_platform(std::move(platform))
    , m_clientId(PlayerCoreClientId)
    , m_playerType("site")
{
    m_headers["Accept"] = AcceptJsonVersion;
    setClientId(PlayerCoreClientId);
}

void TokenHandler::setAuthToken(const std::string& token)
{
    m_authToken = token;

    // Wipe cached playlists using the wrong auth token
    m_tokens.clear();
}

void TokenHandler::setClientId(const std::string& id)
{
    m_clientId = id;
    m_tokens.clear();
}

void TokenHandler::setUniqueId(const std::string& id)
{
    m_headers["Cookie"] = "unique_id=" + id;
}

bool TokenHandler::isPlayerCoreClientId() const
{
    return m_clientId == PlayerCoreClientId;
}

const TokenHandler::TokenResponse& TokenHandler::getToken(const TwitchLink& link)
{
    // look up the cached version if it exists and if it's not expired use it
    if (m_tokens.count(link.getId())) {
        const auto& response = m_tokens[link.getId()];

        std::string err;
        auto json = json11::Json::parse(response.token, err);
        int expires = json["expires"].int_value(); // expire time in unix time

        using namespace std::chrono;
        int unixNow = static_cast<int>(duration_cast<seconds>(system_clock::now().time_since_epoch()).count());
        if (expires && unixNow >= expires) {
            // it's expired remove it
            m_tokens.erase(link.getId());
        } else {
            return response;
        }
    }

    static TokenResponse empty;
    return empty;
}

void TokenHandler::removeToken(const TwitchLink& link)
{
    m_tokens.erase(link.getId());
}

std::string TokenHandler::createAccessTokenRequest(const TwitchLink& link)
{
    if (link.getId().empty()) {
        return "";
    }

    // create API request url
    UriBuilder builder("https", "api.twitch.tv");
    std::string path = "api/";
    path += link.getContentType() == TwitchLink::ContentType::Live ? "channels/" : "vods/";
    path += link.getId();
    path += "/access_token";
    builder.setPath(path);
    builder.setParameter("client_id", m_clientId);

    std::string platform = m_platform->getName();
    auto lower = [](char c) { return std::tolower(c, std::locale()); };
    std::transform(platform.begin(), platform.end(), platform.begin(), lower);

    builder.setParameter("platform", platform);
    builder.setParameter("player_backend", "mediaplayer");
    builder.setParameter("player_type", m_playerType);

    if (!m_authToken.empty()) {
        builder.setParameter("oauth_token", m_authToken);
    }

    return builder.build();
}

const TokenHandler::TokenResponse& TokenHandler::parseTokenResponse(const TwitchLink& link, const std::string& content)
{
    std::string err;
    json11::Json resp = json11::Json::parse(content, err);

    const std::string& token = resp["token"].string_value();
    const std::string& sig = resp["sig"].string_value();

    m_tokens[link.getId()].sig = sig;
    m_tokens[link.getId()].token = token;
    return m_tokens[link.getId()];
}
}
