#include "TwitchLink.hpp"
#include "util/UriBuilder.hpp"
#include <algorithm>
#include <locale>
#include <vector>

namespace twitch {
TwitchLink::TwitchLink(const std::string& url)
{
    auto videoId = parse(url);
    m_id = videoId.first;
    m_contentType = videoId.second;
    UriBuilder::getParametersFromUrl(url, m_queryParameters);
}

std::pair<std::string, TwitchLink::ContentType> TwitchLink::parse(const std::string& url)
{
    // supporting these deep link types:
    // live:  https://twitch.tv/<channel>
    // vods:  https://twitch.tv/<channel>/v/<vodId> and https://twitch.tv/videos/<vodId>
    //        https://twitch.tv/<channel>/v/v<vodId> and https://twitch.tv/videos/v<vodId>
    // clips: https://clips.twitch.tv/<clip>

    std::string path;
    std::string sourceLower = url;
    auto lower = [](char c) { return std::tolower(c, std::locale()); };
    std::transform(sourceLower.begin(), sourceLower.end(), sourceLower.begin(), lower);

    // is it a clip link?
    std::vector<std::string> clipLinks = {
        "http://clips.twitch.tv/",
        "https://clips.twitch.tv/",
    };
    for (const auto& link : clipLinks) {
        if (sourceLower.compare(0, link.length(), link) == 0) {
            path = sourceLower.substr(link.size());
            auto position = path.find_last_of('/');
            if (position != std::string::npos) {
                path = path.substr(0, path.size() - 2);
            }
            return std::pair<std::string, ContentType>(path, ContentType::Clip);
        }
    }

    // is it a TTV link?
    std::vector<std::string> links = {
        "http://twitch.tv/",
        "https://twitch.tv/",
        "http://www.twitch.tv/",
        "https://www.twitch.tv/",
    };
    for (const auto& link : links) {
        if (sourceLower.compare(0, link.length(), link) == 0) {
            path = sourceLower.substr(link.size());
            break;
        }
    }

    if (path.empty()) {
        return std::pair<std::string, ContentType>("", ContentType::Unknown);
    }

    std::vector<std::string> vodPrefixes = {
        "v/",
        "videos/",
    };

    std::string id;
    ContentType contentType = ContentType::Live;

    // check if live channel or vod id
    auto position = path.find_last_of('/');
    if (position == std::string::npos) {
        id = path;
        contentType = ContentType::Live;
    } else if (position == path.size() - 1
        && path.find(vodPrefixes[0]) == std::string::npos
        && path.find(vodPrefixes[1]) == std::string::npos) {
        id = path.substr(0, path.size() - 2);
        contentType = ContentType::Live;
    } else {
        for (const auto& prefix : vodPrefixes) {
            auto pos = path.find(prefix);
            if (pos != std::string::npos) {
                id = path.substr(pos + prefix.size());
                if (id.length() > 0 && id[0] == 'v') {
                    id = id.substr(1);
                }
                contentType = ContentType::VOD;
                break;
            }
        }
    }

    return std::pair<std::string, TwitchLink::ContentType>(id, contentType);
}

bool TwitchLink::isUsherUrl(const std::string& url)
{
    return url.find("://usher.ttvnw.net") != std::string::npos
        || url.find("://usher-staging.internal.justin.tv") != std::string::npos
        || url.find("://usher.staging.justin.tv/") != std::string::npos;
}
}
