#pragma once

#include <map>
#include <string>

namespace twitch {
/**
 * Represents a twitch link, takes a url input parameter, matches the url against the twitch link
 * format:
 * https://twitch.tv/<channelname>
 * https://twitch.tv/<channelname>/v/<vodId> and http://twitch.tv/videos/<vodId>
 * https://clips.twitch.tv/<clip>
 *
 * If the url matches then it's parsed into it's id and content type. If the input url doesn't
 * match the link format the id is empty.
 */
class TwitchLink {
public:
    explicit TwitchLink(const std::string& url);

    const std::string& getId() const { return m_id; };
    enum class ContentType {
        Unknown,
        Live,
        VOD,
        Clip,
    };
    bool isClip() { return m_contentType == ContentType::Clip; }
    bool isChannel() { return m_contentType == ContentType::Live || m_contentType == ContentType::VOD; }
    ContentType getContentType() const { return m_contentType; };
    const std::map<std::string, std::string>& getQueryParameters() const { return m_queryParameters; };
    static std::pair<std::string, ContentType> parse(const std::string& url);
    static bool isUsherUrl(const std::string& url);

private:
    std::string m_id;
    ContentType m_contentType;
    std::map<std::string, std::string> m_queryParameters;
};
}
