#include "UriBuilder.hpp"
#include <cctype>
#include <string>
#include <vector>

namespace twitch {
UriBuilder::UriBuilder(const std::string& scheme, const std::string& host, const std::string& path)
    : m_scheme(scheme)
    , m_host(host)
    , m_path(path)
    , m_port(-1)
{
}

std::string UriBuilder::build()
{
    std::string uri(m_scheme);
    uri += "://";
    uri += m_host;

    if (m_port > 0) {
        uri += ":";
        uri += std::to_string(m_port);
    }

    if (!m_path.empty()) {
        uri += "/";
        uri += m_path;
    }

    if (!m_params.empty()) {
        std::string query;
        for (const auto& i : m_params) {
            if (!query.empty()) {
                query += "&";
            }
            query += i.first;
            query += "=";
            query += i.second;
        }
        uri += "?";
        uri += query;
    }
    return uri;
}

std::string UriBuilder::encode(const std::string& string)
{
    std::string encoded;
    encoded.reserve(string.length() * 3);

    for (const auto c : string) {
        if (std::isalnum(static_cast<unsigned char>(c)) || c == '-' || c == '_' || c == '.' || c == '~') {
            encoded.append(1, c);
        } else {
            static const char dec2Hex[16 + 1] = "0123456789ABCDEF";
            encoded.append(1, '%');
            encoded.append(1, dec2Hex[(c >> 4) & 0x0F]);
            encoded.append(1, dec2Hex[c & 0x0F]);
        }
    }

    return encoded;
}

void UriBuilder::getParametersFromUrl(const std::string& url, std::map<std::string, std::string>& values)
{
    // avoiding istringstream here for code size reasons
    auto queryStart = url.find_last_of('?');

    if (queryStart != std::string::npos && queryStart + 1 < url.size()) {
        std::string parameters = url.substr(queryStart + 1);

        auto split = [](const std::string& str, char delimiter, std::vector<std::string>& parts) {
            size_t i = 0;
            auto pos = str.find(delimiter);
            while (pos != std::string::npos) {
                parts.push_back(str.substr(i, pos - i));
                i = ++pos;
                pos = str.find(delimiter, pos);

                if (pos == std::string::npos)
                    parts.push_back(str.substr(i, str.length()));
            }
        };

        if (!parameters.empty()) {
            std::vector<std::string> parts;
            split(parameters, '&', parts);

            for (const auto& part : parts) {
                std::vector<std::string> keyvalue;
                split(part, '=', keyvalue);
                if (keyvalue.size() == 2) {
                    values[keyvalue[0]] = keyvalue[1];
                }
            }
        }
    }
}

void UriBuilder::setEncodedParameter(const std::string& key, const std::string& value)
{
    m_params[key] = encode(value);
}
}
