#include "Uuid.hpp"
#include "Hex.hpp"
#include "Random.hpp"
#include <algorithm>
#include <memory>

namespace twitch {
Uuid Uuid::fromBytes(const std::vector<uint8_t>& data)
{
    size_t index = 0;
    Uuid uuid {};
    if (data.size() == 16) {
        uuid.timeLow = (uint32_t)((data[0] << 24) | (data[1] << 16) | (data[2] << 8) | data[3]);
        index += 4;
        uuid.timeMid = (uint16_t)((data[index] << 8) | data[index + 1]);
        index += 2;
        uuid.timeHigh = (uint16_t)((data[index] << 8) | data[index + 1]);
        index += 2;
        uuid.clockSeq = (uint16_t)((data[index] << 8) | data[index + 1]);
        index += 2;
        for (size_t i = 0; i < 6; i++) {
            uuid.node[i] = data[index + i];
        }
    }
    return uuid;
}

Uuid Uuid::fromString(const std::string& str)
{
    std::string stripped(str);
    stripped.erase(std::remove(stripped.begin(), stripped.end(), '-'), stripped.end());
    return fromBytes(Hex::decode(stripped.data(), stripped.size()));
}

Uuid Uuid::random()
{
    auto buffer = Random::buffer(16);
    buffer[6] = (buffer[6] & 0x0f) | 0x40; // Version 4 (random)
    buffer[8] = (buffer[8] & 0x3f) | 0x80;
    return fromBytes(buffer);
}

std::vector<uint8_t> Uuid::toBytes() const
{
    std::vector<uint8_t> data;
    data.reserve(16);
    // low
    data.push_back((timeLow >> 24) & 0xFF);
    data.push_back((timeLow >> 16) & 0xFF);
    data.push_back((timeLow >> 8) & 0xFF);
    data.push_back(timeLow & 0xFF);
    // mid
    data.push_back((timeMid >> 8) & 0xFF);
    data.push_back(timeMid & 0xFF);
    // high
    data.push_back((timeHigh >> 8));
    data.push_back(timeHigh & 0xFF);
    // clock
    data.push_back((clockSeq >> 8));
    data.push_back(clockSeq & 0xFF);
    // node
    data.insert(data.end(), node.begin(), node.end());
    return data;
}

std::string Uuid::toString() const
{
    return toString("%02x%02x%02x%02x-%02x%02x-%02x%02x-%02x%02x-%02x%02x%02x%02x%02x%02x");
}

std::string Uuid::toString(const char* format) const
{
    size_t size = printFormat(format, nullptr, 0);
    std::unique_ptr<char[]> buffer(new char[size]);
    printFormat(format, buffer.get(), size);
    return std::string(buffer.get(), buffer.get() + size - 1);
}

size_t Uuid::printFormat(const char* format, char* buffer, size_t size) const
{
    return std::snprintf(buffer, size, format,
        (timeLow >> 24) & 0xFF, (timeLow >> 16) & 0xFF, (timeLow >> 8) & 0xFF, timeLow & 0xFF,
        (timeMid >> 8) & 0xFF, timeMid & 0xFF,
        (timeHigh >> 8) & 0xFF, timeHigh & 0xFF,
        (clockSeq >> 8) & 0xFF, clockSeq & 0xFF,
        node[0], node[1], node[2], node[3], node[4], node[5]);
}

bool Uuid::operator==(const Uuid& other) const
{
    return timeLow == other.timeLow
        && timeMid == other.timeMid
        && timeHigh == other.timeHigh
        && clockSeq == other.clockSeq
        && node == other.node;
}

Uuid operator"" _uuid(const char* s, std::size_t n)
{
    return Uuid::fromString(std::string(s, n));
}
}
