#pragma once
#include "playercore/Player.hpp"
#include "time_util.hpp"
#include "EmptyPlayerListener.hpp"
#include <atomic>
#include <condition_variable>
#include <gtest/gtest.h>
#include <mutex>
#include <type_traits>

namespace twitch {
namespace test {
/**
 * Monitor Player callbacks
 * Note: Only `onSeekCompleted` is currently implemented.
 */
class PlayerCallbackMonitor : public EmptyPlayerListener {
public:
    PlayerCallbackMonitor();
    ~PlayerCallbackMonitor() override = default;

    void onSeekCompleted(twitch::MediaTime time) override;

    /**
     * Wait for the player to receive onSeekCompleted
     *
     * @tparam DURATION is type of std::chrono::duration
     * @param test asynchronous test function to run
     * @param timeout duration the function will at most wait for
     * @returns ::testing::AssertionSuccess if the expected callback was received
     *      before timing out, otherwise ::testing::AssertionFailure
     */
    template <typename DURATION>
    ::testing::AssertionResult waitForSeekCompleted(std::function<void()> test, const DURATION& timeout)
    {
        m_seekCompleted = false;

        test();

        auto predicate = [this]() { return m_seekCompleted.load(); };
        std::unique_lock<std::mutex> lock(m_mutex);
        if (m_condition.wait_for(lock, timeout, predicate)) {
            return ::testing::AssertionSuccess();
        } else {
            return ::testing::AssertionFailure() << "Waiting for onSeekCompleted timed out after " << TimeUtil::toString(timeout) << "(s)";
        }
    }

protected:
    std::atomic<bool> m_seekCompleted;

    mutable std::mutex m_mutex;
    mutable std::condition_variable m_condition;
};
}
}
