#pragma once

#include <functional>
#include <map>
#include <mutex>

namespace twitch {
namespace test {
template <typename T>
class Event;

template <typename R, typename... Args>
class Event<R(Args...)> {
public:
    static const int InvalidHandle = -1;

    Event()
        : m_handle(0)
    {
    }
    virtual ~Event()
    {
        unsubscribeAll();
    }

    void operator()(Args... args) { raise(std::forward<Args>(args)...); }

    int operator+=(std::function<void(Args...)> listener)
    {
        return subscribe(listener);
    }

    bool operator-=(int handle)
    {
        return unsubscribe(handle);
    }

    void unsubscribeAll()
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        m_listeners.clear();
    }

private:
    Event(const Event&) = delete;
    const Event& operator=(const Event&) = delete;

    void raise(Args... args)
    {
        std::lock_guard<std::mutex> lock(m_mutex);

        auto listeners = m_listeners;
        for (auto& listener : listeners) {
            listener.second(args...);
        }
    }

    int subscribe(std::function<void(Args...)> listener)
    {
        if (listener) {
            std::lock_guard<std::mutex> lock(m_mutex);
            m_listeners[++m_handle] = listener;
            return m_handle;
        }

        return InvalidHandle;
    }

    bool unsubscribe(int handle)
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        size_t numErased = m_listeners.erase(handle);
        return numErased > 0;
    }

    std::map<int, std::function<void(Args...)>> m_listeners;
    std::mutex m_mutex;
    int m_handle;
};
}
}
