#pragma once

#include "event.hpp"
#include "debug/trace.hpp"
#include <cassert>
#include <functional>
#include <list>
#include <utility>

namespace twitch {
namespace test {
/**
 * EventHandler provides a templated interface for listening to Event<R(Args...>)
 */
template <typename... Args>
class EventHandler {
public:
    using EventType = Event<void(Args...)>;
    using EventReceivedType = std::tuple<typename std::decay<Args>::type...>;

    virtual ~EventHandler() = default;

protected:
    virtual void onEvent(Args... args) = 0;
};

/**
 * EventSubscriber provides methods for controlling event subscriptions
 */
template <typename... Args>
class EventSubscriber : public EventHandler<Args...> {
public:
    using EventHandlerType = EventHandler<Args...>;
    using EventType = Event<void(Args...)>;

    virtual ~EventSubscriber() = default;

protected:
    int m_handle = EventType::InvalidHandle;
    EventType* m_event = nullptr;

public:
    bool isSubscribed() const
    {
        return m_handle != EventType::InvalidHandle && m_event != nullptr;
    }

    bool subscribe(EventType* event)
    {
        if (isSubscribed()) {
            TRACE_ERROR("EventSubscriber::subscribe(): Must unsubscribe from previous Event before subscribing to another");
            return false;
        }

        m_handle = (*event += [this](Args... args) {
            this->onEvent(args...);
        });

        if (m_handle == EventType::InvalidHandle) {
            TRACE_ERROR("EventSubscriber::subscribe(): Failed due to invalid event handle");
            return false;
        }

        m_event = event;
        return true;
    }

    bool unsubscribe()
    {
        if (!isSubscribed()) {
            TRACE_ERROR("EventSubscriber::unsubscribe(): Not currently subscribed to any events");
            return false;
        }

        bool success = (*m_event -= m_handle);
        if (!success) {
            TRACE_ERROR("EventSubscriber::unsubscribe(): Unsubscribe handle from event failed");
        }

        m_handle = EventType::InvalidHandle;
        m_event = nullptr;
        return success;
    }
};

/**
 * EventList stores its received events in a list
 */
template <typename... Args>
class EventList : public EventHandler<Args...> {
public:
    using EventListHandler = EventHandler<Args...>;
    using EventReceivedType = typename EventListHandler::EventReceivedType;

    ~EventList() override = default;

protected:
    std::list<EventReceivedType> m_events;

public:
    std::list<EventReceivedType> getEvents() { return m_events; }

protected:
    void onEvent(Args... args) override
    {
        m_events.emplace_back(args...);
    }
};

/**
 * EventListener provides methods to listen to and retrieve all received events
 */
template <typename... Args>
class EventListener : public EventList<Args...>, public EventSubscriber<Args...> {
public:
    using EventListType = EventList<Args...>;
    using EventSubscriberType = EventSubscriber<Args...>;
    using EventReceivedType = typename EventSubscriberType::EventReceivedType;

    ~EventListener() override = default;

protected:
    void onEvent(Args... args) override
    {
        EventListType::onEvent(args...);
    }
};
}
}
