#pragma once
#include "event_listener.hpp"
#include <condition_variable>
#include <mutex>
#include "debug/trace.hpp"

namespace twitch {
namespace test {
/**
 * EventCondition provides methods for waiting for an event
 */
template <typename... Args>
class EventCondition : public EventHandler<Args...> {
public:
    using EventHandlerType = EventHandler<Args...>;
    using EventPredicateType = std::function<bool(Args...)>;
    using EventReceivedType = typename EventHandlerType::EventReceivedType;

protected:
    std::condition_variable m_condition;
    std::mutex m_mutex;
    EventPredicateType m_predicate;

public:
    template <typename DURATION>
    bool waitFor(DURATION timeout, EventReceivedType& received)
    {
        EventPredicateType predicate = [&received](Args... params) mutable -> bool {
            received = EventReceivedType(params...);
            return true;
        };

        return waitFor(timeout, predicate);
    }

    template <typename DURATION>
    bool waitFor(DURATION timeout, EventPredicateType predicate)
    {
        std::unique_lock<std::mutex> lock(m_mutex);
        if (m_predicate) {
            TRACE_ERROR("Already waiting for another event");
            return false;
        }

        m_predicate = predicate;
        bool success = m_condition.wait_for(lock, timeout) == std::cv_status::no_timeout;
        m_predicate = nullptr;
        return success;
    }

protected:
    void onEvent(Args... args) override
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        if (m_predicate) {
            if (m_predicate(args...)) {
                m_condition.notify_one();
            }
        }
    }
};

/**
 * EventMonitor combines the EventCondition with EventSubscriber
 */
template <typename... Args>
class EventMonitor : public EventCondition<Args...>, public EventSubscriber<Args...> {
public:
    using EventConditionType = EventCondition<Args...>;
    using EventSubscriberType = EventSubscriber<Args...>;
    using EventReceivedType = typename EventSubscriberType::EventReceivedType;

protected:
    void onEvent(Args... args) override
    {
        EventConditionType::onEvent(args...);
    }
};
}
}
