#include "player_error_monitor.hpp"

namespace twitch {
namespace test {
using namespace std::placeholders;
std::list<std::tuple<twitch::ErrorSource, twitch::MediaResult, std::string>> PlayerErrorMonitor::getErrors()
{
    std::lock_guard<std::mutex> lock(m_mutex);
    return m_errors;
}

void PlayerErrorMonitor::onError(const Error& error)
{
    std::lock_guard<std::mutex> lock(m_mutex);
    m_errors.emplace_back(error.source, error.result, error.message);
    m_condition.notify_one();
}

::testing::AssertionResult PlayerErrorMonitor::waitFor(twitch::ErrorSource source, twitch::MediaResult result, const std::string& message, std::function<void()> test, std::chrono::microseconds timeout)
{
    const auto predicate = [&]() {
        if (m_errors.empty()) {
            return false;
        } else {
            const auto& last = m_errors.back();
            return std::get<0>(last) == source && std::get<1>(last) == result && std::get<2>(last) == message;
        }
    };

    test();

    std::unique_lock<std::mutex> lock(m_mutex);

    if (m_condition.wait_for(lock, timeout, predicate)) {
        return testing::AssertionSuccess();
    } else {
        return testing::AssertionFailure() << "Monitor timed out waiting for error=" << errorSourceString(source)
                                           << ", result=" << mediaResultString(result) << ", message=" << message;
    }
}
}
}
