#pragma once
#include "playercore/Player.hpp"
#include "debug/trace.hpp"
#include <cassert>
#include <map>
#include <set>

namespace twitch {
namespace test {
template <typename T>
class StateMachine {
public:
    // Map of state transitions from a given state
    using StateTransitions = std::map<T, std::set<T>>;

    StateMachine(T currentState, const StateTransitions& validTransitions)
        : m_current(currentState)
        , m_validTransitions(validTransitions)
    {
    }

    static const char* stateToString(T state);

    T getCurrentState() const { return m_current; }

    // @returns true if the state change is valid, otherwise false
    bool onStateChange(T state)
    {
        auto itr = m_validTransitions.find(m_current);

        if (itr == m_validTransitions.end()) {
            // we are not even allowed to be in this state
            return false;
        }

        const auto& validStates = itr->second;

        bool validTransition = validStates.find(state) != validStates.end();
        if (!validTransition) {
            TRACE_ERROR("Invalid state change from %s to %s", stateToString(m_current), stateToString(state));
        }

        m_current = state;
        return validTransition;
    }

protected:
    T m_current;
    StateTransitions m_validTransitions;
};

class PlayerStateMachine : public StateMachine<Player::State> {
public:
    using StateMachine = StateMachine<Player::State>;
    static StateTransitions ValidLiveStateTransitions;
    static StateTransitions ValidVODStateTransitions;

    PlayerStateMachine(Player::State current, bool isLive);
};
}
}
