#pragma once
#include "util/Concurrency.hpp"
#include "playercore/Player.hpp"
#include "time_util.hpp"
#include <future>
#include <gtest/gtest.h>
#include <type_traits>

namespace twitch {
namespace test {
class PlayerTimeoutMonitor {
public:
    PlayerTimeoutMonitor() {}

    template <typename DURATION>
    ::testing::AssertionResult waitFor(std::function<void()> test, const DURATION& timeout)
    {
        m_testStarted = false;
        m_testCompleted = false;
        auto future = std::async(std::launch::async,
            [this, test] {
                m_testStarted = true;
                m_condition.notify_one();
                test();
                {
                    std::unique_lock<std::mutex> lock(m_mutex);
                    m_testCompleted = true;
                }
                m_condition.notify_one();
            });

        while (!m_testStarted) {
            std::unique_lock<std::mutex> lock(m_mutex);
            if (m_condition.wait_for(lock, timeout) == std::cv_status::timeout) {
                return ::testing::AssertionFailure() << " timed out after " << TimeUtil::toString(timeout) << "(s)";
            }
        }

        std::unique_lock<std::mutex> lock(m_mutex);
        if (!m_condition.wait_for(lock, timeout, [this]() { return m_testCompleted; })) {
            return ::testing::AssertionFailure() << " timed out after " << TimeUtil::toString(timeout) << "(s)";
        }

        future.get();
        return ::testing::AssertionSuccess();
    }

private:
    bool m_testStarted;
    bool m_testCompleted;
    mutable std::mutex m_mutex;
    mutable std::condition_variable m_condition;
};
}
}
