#pragma once
#include <condition_variable>
#include <functional>
#include <gtest/gtest.h>
#include <playercore/Quality.hpp>
#include <mutex>

namespace twitch {
namespace test {
/**
 * Monitor the quality changes reported by a Player instance
 */
class QualityMonitor {
public:
    explicit QualityMonitor(const twitch::Quality& quality);
    ~QualityMonitor() = default;

    /**
     * Wait for the quality to change to a given quality name
     *
     * @tparam DURATION specifies the type of std::chrono::duration
     * @param qualityName name of quality
     * @param test function to execute for this test
     * @param timeout duration the function will at most wait for
     * @return ::testing::AssertionSuccess if the expected quality was reached
     *      before timing out, otherwise ::testing::AssertionFailure
     */
    template <typename DURATION>
    ::testing::AssertionResult waitFor(const std::string& qualityName, std::function<void()> test, const DURATION& timeout)
    {
        const auto predicate = [&]() { return m_quality.name == qualityName; };
        {
            std::lock_guard<std::mutex> lock(m_mutex);

            if (predicate()) {
                return ::testing::AssertionFailure() << "Current quality is already: " << qualityName;
            }
        }

        test();

        std::unique_lock<std::mutex> lock(m_mutex);

        if (m_condition.wait_for(lock, timeout, predicate)) {
            return testing::AssertionSuccess();
        } else {
            return testing::AssertionFailure() << "Monitor timed out waiting for quality: " << qualityName;
        }
    }

    void onQualityChanged(const twitch::Quality& quality);

protected:
    twitch::Quality m_quality;

    std::mutex m_mutex;
    std::condition_variable m_condition;
};
}
}
