#pragma once
#include "playercore/Player.hpp"
#include "time_util.hpp"
#include <condition_variable>
#include <gtest/gtest.h>
#include <list>
#include <mutex>
#include <type_traits>

namespace twitch {
namespace test {
/**
 * Counts the number of times the Player's State has changed
 */
class PlayerStateChangeMonitor {
public:
    PlayerStateChangeMonitor(twitch::Player::State state);
    ~PlayerStateChangeMonitor() = default;

public:
    void onStateChanged(twitch::Player::State state);

    /**
     * Wait for the player state to change to an ordered list of expected states
     *
     * @tparam DURATION is type of std::chrono::duration
     * @param states ordered list of expected player states
     * @param test asynchronous test function to run
     * @param timeout duration the function will at most wait for
     * @returns ::testing::AssertionSuccess if the expected state was reached
     *      before timing out, otherwise ::testing::AssertionFailure
     */
    template <typename DURATION>
    ::testing::AssertionResult waitFor(std::list<twitch::Player::State> states, std::function<void()> test, const DURATION& timeout)
    {
        // Find the last state before running the test
        std::list<twitch::Player::State>::iterator actual;
        {
            std::lock_guard<std::mutex> lock(m_mutex);
            actual = m_states.end();
            if (!m_states.empty()) {
                --actual;

                if (*actual == states.front()) {
                    return ::testing::AssertionFailure() << "Already in state=" << twitch::Player::stateToString(states.front());
                }
            }
        }

        test();

        auto expected = states.begin();
        {
            std::unique_lock<std::mutex> lock(m_mutex);

            // We don't actually care about the exact state before the test started, only the states
            // after it
            if (actual != m_states.end()) {
                ++actual;
            }

            // Increment the expected and actual state iterators until all of the expected states
            // have been received
            while (expected != states.end()) {
                // Check to see if we have a new state to verify, otherwise, wait for a new state
                if (actual != m_states.end()) {
                    if (*expected == *actual) {
                        ++expected;
                        ++actual;
                    } else {
                        return ::testing::AssertionFailure() << "Expected=" << twitch::Player::stateToString(*expected) << " Actual=" << twitch::Player::stateToString(*actual);
                    }
                } else if (m_condition.wait_for(lock, timeout) == std::cv_status::timeout) {
                    return ::testing::AssertionFailure() << "Waiting for state=" << twitch::Player::stateToString(*expected) << " timed out after " << TimeUtil::toString(timeout) << "(s)";
                } else {
                    --actual;
                }
            }
        }

        return ::testing::AssertionSuccess();
    }

    /**
     * Wait for the player state to change to a given state
     *
     * @tparam DURATION is type of std::chrono::duration
     * @param state desired or expected player state to deem the test successful
     * @param test asynchronous test function to run
     * @param timeout duration the function will at most wait for
     * @returns ::testing::AssertionSuccess if the expected state was reached
     *      before timing out, otherwise ::testing::AssertionFailure
     */
    template <typename DURATION>
    ::testing::AssertionResult waitFor(twitch::Player::State state, std::function<void()> test, const DURATION& timeout)
    {
        {
            std::lock_guard<std::mutex> lock(m_mutex);
            if (!m_states.empty() && m_states.back() == state) {
                return ::testing::AssertionFailure() << "Already in state=" << twitch::Player::stateToString(state);
            }
        }

        test();

        auto predicate = [this, state]() { return m_states.back() == state; };
        std::unique_lock<std::mutex> lock(m_mutex);
        if (m_condition.wait_for(lock, timeout, predicate)) {
            return ::testing::AssertionSuccess();
        } else {
            return ::testing::AssertionFailure() << "Waiting for state=" << twitch::Player::stateToString(state) << " timed out after " << TimeUtil::toString(timeout) << "(s)";
        }
    }

    /**
     * Wait for a certain amount of time, make sure the state does *NOT* change from the current state
     * @param the current state of the player and the state we should be staying into
     * @param test asynchronous test function to run
     * @param timeout duration the function will wait for
     */
    template <typename DURATION>
    ::testing::AssertionResult waitWhileStayingInState(twitch::Player::State state, std::function<void()> test, const DURATION& timeout)
    {
        {
            std::lock_guard<std::mutex> lock(m_mutex);
            if (m_states.empty() || m_states.back() != state) {
                return ::testing::AssertionFailure() << "Not in expected state=" << twitch::Player::stateToString(state);
            }
        }

        test();

        auto predicate = [this, state]() { return m_states.back() != state; }; // give up as soon as we change to a different state
        std::unique_lock<std::mutex> lock(m_mutex);
        if (m_condition.wait_for(lock, timeout, predicate)) {
            return ::testing::AssertionFailure() << "Changed to state=" << twitch::Player::stateToString(m_states.back()) << " instead of staying in state=" << twitch::Player::stateToString(state) << " during wait of " << TimeUtil::toString(timeout) << "(s)";
        } else {
            return ::testing::AssertionSuccess();
        }
    }

protected:
    std::list<twitch::Player::State> m_states;

    mutable std::mutex m_mutex;
    mutable std::condition_variable m_condition;
};
}
}
