#pragma once
#include <cassert>
#include <chrono>
#include <string>
#include <type_traits>

namespace twitch {
namespace test {
/**
 * Time helper functions for anything related to time
 */
namespace TimeUtil {
template <typename DURATION>
std::string getPrefix()
{
    using namespace std::chrono;
    if (std::is_same<nanoseconds, DURATION>::value) {
        return "nsec";
    } else if (std::is_same<microseconds, DURATION>::value) {
        return "usec";
    } else if (std::is_same<milliseconds, DURATION>::value) {
        return "msec";
    } else if (std::is_same<seconds, DURATION>::value) {
        return "sec";
    } else if (std::is_same<minutes, DURATION>::value) {
        return "min";
    } else if (std::is_same<hours, DURATION>::value) {
        return "hrs";
    }

    assert(false);
    return "";
}

/**
 * @returns true if duration1 and duration2 are within epsilon, otherwise false
 */
template <typename DURATION1, typename DURATION2, typename DURATION3>
bool isNear(DURATION1 duration1, DURATION2 duration2, DURATION3 epsilon)
{
    using DurationBaseType = std::chrono::nanoseconds;

    auto duration1Nano = std::chrono::duration_cast<DurationBaseType>(duration1);
    auto duration2Nano = std::chrono::duration_cast<DurationBaseType>(duration2);
    auto epsilonNano = std::chrono::duration_cast<DurationBaseType>(epsilon);

    auto durationDiff = (duration1 <= duration2) ? duration2Nano - duration1Nano : duration1Nano - duration2Nano;

    assert(durationDiff.count() >= 0);
    return durationDiff.count() < epsilonNano.count();
}

/**
 * @params duration the duration to to format
 * @returns the duration string formatted as "${duration.count()} ${duration prefix}"
 */
template <typename DURATION>
std::string toString(const DURATION& duration)
{
    return std::to_string(duration.count()) + " " + getPrefix<DURATION>();
}

/**
 * @tparam DURATION_OUT the duration type that we should cast to
 * @params duration the duration to to format
 * @returns the duration string formatted as "${duration.count()} ${duration prefix}"
 */
template <typename DURATION_OUT, typename DURATION_IN>
std::string toString(const DURATION_IN& duration)
{
    return toString(std::chrono::duration_cast<DURATION_OUT>(duration));
}
};
}
}
