#pragma once
#include <cassert>
#include <chrono>
#include <functional>
#include <gtest/gtest.h>

/**
 * TimedTestRunner will block on run() up to the given duration and invoke a testing callback
 * on a regularly scheduled interval
 */
namespace twitch {
namespace test {
class TimedTestRunner {
public:
    using clock_t = std::chrono::steady_clock;
    using time_t = std::chrono::steady_clock::duration;

    /**
     * @param duration how long the test should run for
     * @param period interval in which the test function should be invoked to 
     */
    template <typename DURATION, typename PERIOD>
    TimedTestRunner(const DURATION& duration, const PERIOD& period)
        : m_duration(std::chrono::duration_cast<time_t>(duration))
        , m_period(std::chrono::duration_cast<time_t>(period))
    {
        assert(m_duration.count() > m_period.count());
        assert(getNumIntervals() > 1);
    }

    const time_t& getTotalTestTime() const;

    /**
     * Run the test runner
     *
     * @params test function that is invoked every test period.  If any test fails,
     *      then the test will stop running
     * @returns true if all of its tests passed, otherwise false
     */
    bool run(std::function<::testing::AssertionResult(time_t)> test);

protected:
    time_t m_duration;
    time_t m_period;

    // The total time the test has run where 0 represents the beginning of the test
    time_t m_totalTestTime{ 0 };

protected:
    size_t getNumIntervals() const;
};
}
}
