#include "PlayerTestCase.hpp"

namespace twitch {
namespace test {
const float DefaultTimeoutSeconds = 10;

PlayerTestCase::PlayerTestCase(const std::string& name,
    TestEnvironment& environment,
    std::shared_ptr<twitch::Scheduler> scheduler)
    : TestCase(name, std::move(scheduler))
    , m_environment(environment)
    , m_player(environment.createPlayer(*this))
{
}

PlayerTestCase::~PlayerTestCase()
{
    m_environment.tearDown();
}

PlayerTestCase& PlayerTestCase::action(std::function<void(PlayerTestCase& test)> func)
{
    TestCase::next([func](TestCase& test) {
        func(static_cast<PlayerTestCase&>(test));
    });
    return *this;
}

PlayerTestCase& PlayerTestCase::awaitState(Player::State state)
{
    await([this, state](TestCase& test) {
        onStateChangedEvent += [this, state, &test](Player::State s) {
            onStateChangedEvent.unsubscribeAll();
            if (s == state) {
                resume();
            } else {
                std::string name = Player::stateToString(state);
                std::string message = "Failed to reach state " + name + " within the timeout";
                test.fail(message);
            }
        };
    },
        DefaultTimeoutSeconds);
    return *this;
}

PlayerTestCase& PlayerTestCase::awaitError(const Error& error)
{
    await([this, error](TestCase& test) {
        onErrorEvent += [this, error, &test](const Error& e) {
            onErrorEvent.unsubscribeAll();
            if (e.result == error.result && e.source == error.source) {
                resume();
            } else {
                test.fail("Unexpected error");
            }
        };
    },
        DefaultTimeoutSeconds);
    return *this;
}

PlayerTestCase& PlayerTestCase::awaitQualityChange(const Quality& quality)
{
    await([this, quality](TestCase& test) {
        onQualityChangedEvent += [this, quality, &test](const Quality& q) {
            onQualityChangedEvent.unsubscribeAll();
            if (q == quality) {
                resume();
            } else {
                test.fail("Unexpected quality change");
            }
        };
    },
        DefaultTimeoutSeconds);
    return *this;
}

PlayerTestCase& PlayerTestCase::awaitSeeked()
{
    await([this](TestCase&) {
        onSeekedEvent += [this](MediaTime time) {
            (void) time;
            onSeekedEvent.unsubscribeAll();
            resume();
        };
    },
        DefaultTimeoutSeconds);
    return *this;
}

void PlayerTestCase::onDurationChanged(MediaTime duration)
{
    onDurationChangedEvent(duration);
}

void PlayerTestCase::onError(const Error& error)
{
    onErrorEvent(error);
}

void PlayerTestCase::onMetadata(const std::string& type, const std::vector<uint8_t>& data)
{
    onMetadataEvent(type, data);
}

void PlayerTestCase::onQualityChanged(const Quality& quality)
{
    onQualityChangedEvent(quality);
}

void PlayerTestCase::onRebuffering()
{
    onRebufferingEvent();
}

void PlayerTestCase::onRecoverableError(const Error& error)
{
    (void)error;
}

void PlayerTestCase::onSeekCompleted(MediaTime time)
{
    onSeekedEvent(time);
}

void PlayerTestCase::onSessionData(const std::map<std::string, std::string>& properties)
{
    (void)properties;
}

void PlayerTestCase::onStateChanged(Player::State state)
{
    onStateChangedEvent(state);
}

void PlayerTestCase::onAnalyticsEvent(const std::string& name, const std::string& properties)
{
    onTrackingEvent(name, properties);
}

void PlayerTestCase::onPositionChanged(MediaTime position)
{
    (void)position;
}
}
}
