#include "TestCase.hpp"
#include "TestLog.hpp"

namespace twitch {
namespace test {
TestCase::TestCase(std::string name, std::shared_ptr<Scheduler> scheduler)
    : ScopedScheduler(std::move(scheduler))
    , m_name(std::move(name))
    , m_step(0)
    , m_success(true)
    , m_listener(nullptr)
{
}

TestCase::~TestCase()
{
    cancel();
}

TestCase& TestCase::describe(const std::string& description)
{
    m_description += description;
    return *this;
}

TestCase& TestCase::next(std::function<void(TestCase& test)> func)
{
    await(std::move(func), 0);
    return *this;
}

TestCase& TestCase::await(std::function<void(TestCase& test)> func, float timeout)
{
    m_steps.emplace_back();
    m_steps.back().func = std::move(func);
    m_steps.back().timeout = timeout;
    return *this;
}

void TestCase::assertTrue(bool condition, const std::string& message)
{
    if (!condition) {
        fail(message);
    }
}

void TestCase::assertFalse(bool condition, const std::string& message)
{
    if (condition) {
        fail(message);
    }
}

void TestCase::fail(const std::string& message)
{
    TestLog log;
    log.error("Test %s at step %d message: %s", m_name.c_str(), m_step, message.c_str());
    cancel();
    m_success = false;
}

bool TestCase::isSuccess() const
{
    return m_success;
}

const std::string& TestCase::getDescription() const
{
    return m_description;
}

std::chrono::milliseconds TestCase::getElapsed() const
{
    return std::chrono::duration_cast<std::chrono::milliseconds>(m_end - m_start);
}

void TestCase::run(Listener* listener)
{
    m_step = 0;
    m_listener = listener;
    m_start = std::chrono::system_clock::now();
    if (m_listener) {
        m_listener->onTestCaseStart(m_name);
    }

    nextStep();
}

void TestCase::resume()
{
    if (m_step >= m_steps.size()) {
        auto next = m_steps[m_step];
        // cancel timeout if present
        if (next.timeoutCancel) {
            next.timeoutCancel->cancel();
        }
        // remove this step and run the check
    }
    nextStep();
}

void TestCase::nextStep()
{
    if (m_step >= m_steps.size()) {
        endTest();
    } else {
        auto next = m_steps[m_step];
        if (next.timeout > 0) {
            std::chrono::microseconds timeout(static_cast<long>(next.timeout * std::micro::den));
            next.timeoutCancel = schedule([this]() {
                fail("await timed out");
                endTest();
            },
                timeout);
        } 
        // remove this step and run the check
        m_step++;
        next.func(*this);
        if (m_success) {
            nextStep();
        }
    }
}

void TestCase::endTest()
{
    cancel();

    m_end = std::chrono::system_clock::now();

    //std::chrono::milliseconds duration = std::chrono::duration_cast<std::chrono::milliseconds>(m_end - m_start);
    if (m_listener) {
        m_listener->onTestCaseEnd(m_name, m_success);
    }
}
}
}
