#include "TestRunner.hpp"

namespace twitch {
namespace test {
TestRunner::TestRunner(Listener& listener)
    : m_listener(listener)
    , m_running(false)
    , m_cancelled(false)
    , m_success(true)
{
}

void TestRunner::addTest(const std::shared_ptr<TestCase>& test)
{
    m_testCases.push_back(test);
}

void TestRunner::onTestCaseStart(const std::string& name)
{
    m_log.info("START TEST %s", name.c_str());
}

void TestRunner::onTestCaseEnd(const std::string& name, bool success)
{
    m_log.info("END TEST %s %s", name.c_str(), success ? "PASS" : "FAIL");

    // update overall run result
    if (!success) {
        m_success = false;
    }

    // run the next case if it exists otherwise end
    if (!m_queue.empty() && !m_cancelled) {
        m_queue.pop();
        if (!m_queue.empty()) {
            auto next = m_queue.front();
            m_queue.pop();
            if (next) {
                next->run(this);
            }
        }
    } else {
        m_listener.onTestRunEnd(m_success);
        m_running = false;
    }
}

void TestRunner::run()
{
    m_log.info("START TEST RUNNER...");
    auto empty = std::queue<std::shared_ptr<TestCase>>();
    m_queue.swap(empty);
    for (const auto& test : m_testCases) {
        m_queue.emplace(test);
    }

    m_listener.onTestRunStart();

    if (!m_queue.empty()) {
        auto next = m_queue.front();
        if (next) {
            next->run(this);
        }
    } else {
        m_listener.onTestRunEnd(m_success);
    }
}

void TestRunner::cancel()
{
    m_cancelled = true;
}
}
}
