#include "playerstatetest.hpp"
#include "test/common/testenv/testenv.hpp"
#include "debug/PrintLog.hpp"
#include "PlayerTestCase.hpp"
#include "PlayerTestSuite.hpp"
#include "TestRunner.hpp"

namespace twitch {
namespace test {
static const std::string LiveTestUrl = "https://www.twitch.tv/bufferbunny";
static const std::string VodTestUrl = "https://www.twitch.tv/videos/164446609";

std::unique_ptr<PlayerTestSuite> stateTests(TestEnvironment& environment)
{
    auto suite = std::unique_ptr<PlayerTestSuite>(new PlayerTestSuite(environment));

    // ready on load
    suite->test("ReadyOnLoad")
        .action([](PlayerTestCase& test) {
            test.getPlayer().load(LiveTestUrl);
        })
        .awaitState(Player::State::Ready);

    // idle on stop
    suite->test("IdleOnStop")
        .action([](PlayerTestCase& test) {
            test.getPlayer().load(LiveTestUrl);
        })
        .awaitState(Player::State::Ready)
        .action([](PlayerTestCase& test) {
            test.getPlayer().pause();
        })
        .awaitState(Player::State::Idle);

    // buffer on start
    suite->test("BufferOnStart")
        .action([](PlayerTestCase& test) {
            test.getPlayer().load(LiveTestUrl);
        })
        .awaitState(Player::State::Ready)
        .action([](PlayerTestCase& test) {
            test.getPlayer().play();
        })
        .awaitState(Player::State::Buffering);

    // playing on start
    suite->test("PlayingOnStartAfterBuffer")
        .action([](PlayerTestCase& test) {
            test.getPlayer().load(LiveTestUrl);
        })
        .awaitState(Player::State::Ready)
        .action([](PlayerTestCase& test) {
            test.getPlayer().play();
        })
        .awaitState(Player::State::Buffering)
        .awaitState(Player::State::Playing);

    // playing after pause
    suite->test("PlayingAfterPause")
        .action([](PlayerTestCase& test) {
            test.getPlayer().load(LiveTestUrl);
            test.getPlayer().play();
        })
        .awaitState(Player::State::Playing)
        .action([](PlayerTestCase& test) {
            test.getPlayer().pause();
        })
        .awaitState(Player::State::Idle)
        .action([](PlayerTestCase& test) {
            test.getPlayer().play();
        })
        .awaitState(Player::State::Playing);

    // buffer on seek
    suite->test("BufferOnSeekOutsideBuffer")
        .action([](PlayerTestCase& test) {
            test.getPlayer().load(VodTestUrl);
            test.getPlayer().play();
        })
        .awaitState(Player::State::Playing)
        .action([](PlayerTestCase& test) {
            test.getPlayer().pause();
            twitch::MediaTime offset(std::chrono::milliseconds(500));
            auto time = test.getPlayer().getBufferedPosition() + offset;
            test.getPlayer().seekTo(time);
        })
        .awaitState(Player::State::Buffering);

    // ended at end of stream
    suite->test("EndedAtEndOfStream")
        .action([](PlayerTestCase& test) {
            test.getPlayer().load(VodTestUrl);
            test.getPlayer().play();
        })
        .awaitState(Player::State::Playing)
        .action([](PlayerTestCase& test) {
            twitch::MediaTime offset(std::chrono::seconds(1));
            auto time = test.getPlayer().getDuration() - offset;
            test.getPlayer().seekTo(time);
        })
        .awaitState(Player::State::Ended);

    return suite;
}

}
}
