#include "TestOptions.hpp"
#include "optionparser.hpp"
#include <cctype>

namespace twitch {

enum OptionIndex {
    Unknown,
    Help,
    LogLevel,
    StreamType,
    URL
};

#define DEFAULT_CHANNEL_URL "https://www.twitch.tv/bufferbunny"

const option::Descriptor usage[] = {
    { Unknown, 0, "", "", option::Arg::None, "Usage: IntegrationTest [--gtest_*] [--channel-type=live|VOD] URL\n\nOptions:" },
    { Help, 0, "h", "help", option::Arg::None, "  -h, --help  \tPrint usage and exit" },
    { LogLevel, 0, "", "log-level", option::Arg::Optional, "  --log-level=<level>  \tSpecify logging level" },
    { StreamType, 0, "", "stream-type", option::Arg::Optional, "  --stream-type=live|VOD  \tSpecify if the stream is Live or VOD (optional for Twitch URLs)" },
    { URL, 0, "", "", option::Arg::Optional, "  URL  \tStream URL to test (default is " DEFAULT_CHANNEL_URL ")" },
    { 0, 0, 0, 0, 0, 0 }
};

std::string trim(const std::string& str,
                 const std::string& whitespace = " \t")
{
    const auto strBegin = str.find_first_not_of(whitespace);
    if (strBegin == std::string::npos)
    return ""; // no content
    
    const auto strEnd = str.find_last_not_of(whitespace);
    const auto strRange = strEnd - strBegin + 1;
    
    return str.substr(strBegin, strRange);
}

std::string reduce(const std::string& str,
                   const std::string& fill = " ",
                   const std::string& whitespace = " \t")
{
    // trim first
    auto result = trim(str, whitespace);
    
    // replace sub ranges
    auto beginSpace = result.find_first_of(whitespace);
    while (beginSpace != std::string::npos)
    {
        const auto endSpace = result.find_first_not_of(whitespace, beginSpace);
        const auto range = endSpace - beginSpace;
        
        result.replace(beginSpace, range, fill);
        
        const auto newStart = beginSpace + fill.length();
        beginSpace = result.find_first_of(whitespace, newStart);
    }
    
    return result;
}

bool TestOptions::parse(int argc, char** argv)
{
    // skip program name argv[0] if present
    if (argc > 0) {
        argc--;
        argv++;
    }

    option::Stats stats(usage, argc, argv);
    std::vector<option::Option> options(stats.options_max);
    std::vector<option::Option> buffer(stats.buffer_max);
    option::Parser parse(usage, argc, argv, &options[0], &buffer[0]);

    if (parse.error()) {
        option::printUsage(std::cerr, usage);
        return false;
    }

    int nonOptionsCount = parse.nonOptionsCount();
    if (nonOptionsCount == 0) {
        m_url = DEFAULT_CHANNEL_URL;
    } else if (nonOptionsCount >= 1) {
        // URL is espected to be the first non option
        m_url = parse.nonOption(0);
    }
    // Remove any leading/trailing whitespaces from URL if present
    m_url = reduce(trim(m_url), "", "\"'");
    
    // Output URL
    fprintf(stdout, "Test URL: %s\n", m_url.c_str());
    fflush(stdout);
    
    for (int i = 0; i < parse.optionsCount(); ++i) {
        option::Option& opt = buffer[i];

        switch (opt.index()) {
        case OptionIndex::Help:
            break;
        case OptionIndex::LogLevel:
            if (!setLogLevel(opt.arg)) {
                return false;
            }
            break;
        case OptionIndex::StreamType:
            if (!setStreamType(opt.arg)) {
                return false;
            }
            break;
        default:
            std::cerr << "Unknown option: " << opt.index() << std::endl;
            break;
        }
    }

    if (options[Help]) {
        option::printUsage(std::cout, usage);
        return false;
    }

    return true;
}

auto lower = [](char c) { return std::tolower(c, std::locale()); };

bool TestOptions::setLogLevel(const std::string& logLevel)
{
    std::string lowercased(logLevel);
    std::transform(logLevel.begin(), logLevel.end(), lowercased.begin(), lower);

    if (lowercased == "debug") {
        m_logLevel = Log::Level::Debug;
    } else if (lowercased == "info") {
        m_logLevel = Log::Level::Debug;
    } else if (lowercased == "warning") {
        m_logLevel = Log::Level::Warning;
    } else if (lowercased == "error") {
        m_logLevel = Log::Level::Error;
    } else {
        std::cerr << "Unrecognized log level: " << logLevel << std::endl;
        return false;
    }

    return true;
}

bool TestOptions::setStreamType(const std::string& streamType)
{
    std::string lowercased(streamType);
    std::transform(streamType.begin(), streamType.end(), lowercased.begin(), lower);

    if (lowercased == "live") {
        m_streamType = StreamType::Live;
    } else if (lowercased == "vod") {
        m_streamType = StreamType::VOD;
    } else {
        std::cerr << "Unrecognized stream type: " << streamType << std::endl;
        return false;
    }

    return true;
}

}
