#include "TestStreamConfig.hpp"
#include "IntegrationTest.hpp"
#include "player_test_timeout.hpp"
#include "player/MediaPlayer.hpp"
#include "player_listener.hpp"
#include "state_change_monitor.hpp"
#include "gtestenv.hpp"
#include "debug/TraceLog.hpp"
#include <algorithm>
#include <cassert>

namespace twitch {
namespace test {
using namespace ::testing;

TestStreamConfig::Quality::Quality(const twitch::Quality& quality)
    : twitch::Quality(quality)
{
}

TestStreamConfig::TestStreamConfig(const std::string& url)
    : url(url)
{
}

TestStreamConfig::Quality TestStreamConfig::getQuality(const std::string& name) const
{
    const auto& itr = std::find_if(expectedQualities.begin(), expectedQualities.end(), [&name](const Quality& quality) {
        return quality.name == name;
    });

    return itr == expectedQualities.end() ? TestStreamConfig::Quality() : *itr;
}

AssertionResult TestStreamConfig::init()
{
    if (isInitialized) {
        return AssertionSuccess();
    }

    TraceLog::get().setLevel(IntegrationTest::environment->options().getLogLevel());

    PlayerEventListener eventListener;
    auto player = GTestEnvironment::current->createPlayer(eventListener);

    PlayerStateChangeMonitor stateMonitor(player->getState());
    eventListener.onStateChangedEvent += std::bind(&PlayerStateChangeMonitor::onStateChanged, &stateMonitor, std::placeholders::_1);

    auto isPlaying = stateMonitor.waitFor(twitch::Player::State::Playing, [this, player]() {
        player->load(url);
        player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start); // << ;

    if (!isPlaying) {
        return AssertionFailure() << "Could not load test stream: " << url;
    }

    const auto& qualities = player->getQualities();
    if (qualities.empty()) {
        return AssertionFailure() << "Could not get Qualities for: " << url;
    }

    defaultQuality = Quality();

    for (const auto& quality : qualities) {
        Quality expected(quality);
        expectedQualities.push_back(expected);

        if (quality.isDefault) {
            if (defaultQuality.name.empty()) {
                defaultQuality = expected;
            } else {
                return AssertionFailure() << "Multiple default qualities found for: " << url;
            }
        }
    }

    if (defaultQuality.name.empty()) {
        return AssertionFailure() << "Default quality not found for: " << url;
    }

    isInitialized = true;
    return AssertionSuccess();
}
}
}
