#include "playercore/platform/android/AndroidPlatform.hpp"
#include "testenv.hpp"
#include "IntegrationTest.hpp"
#include <android/log.h>
#include <cstdio>
#include <cstdlib>
#include <jni.h>
#include "jni_ref.hpp"
#include "PlatformJNI.hpp"

extern "C" {
static int pipefd[2] = {};
jobject g_surface = NULL;
jobject g_context = NULL;


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
    (void)reserved;
    twitch::AndroidPlatform::initialize(vm);
    return JNI_VERSION_1_6;
}

static void readfd()
{
    ssize_t size;
    const size_t BufferSize = 256;
    char buffer[BufferSize];
    while ((size = read(pipefd[0], buffer, BufferSize - 1)) > 0) {
        if (buffer[size - 1] == '\n') {
            buffer[size - 1] = 0;
        }
        __android_log_write(ANDROID_LOG_INFO, "Twitch", buffer);
    }
}

void pipestd()
{
    // android sends stdout to dev/null so has to be piped to another fd and read onto the log
    setvbuf(stdout, 0, _IOLBF, 0);
    setvbuf(stderr, 0, _IOLBF, 0);
    pipe(pipefd);
    dup2(pipefd[1], 1);
    dup2(pipefd[1], 2);
    std::thread reader(readfd);
    reader.detach();
}

jint Java_tv_twitch_android_player_test_TestActivity_runTests(JNIEnv* env, jobject activity, jobject context, jobject surface, jstring url, jstring filter, jstring xml)
{
    (void) activity;
    if (g_surface) {
        env->DeleteGlobalRef(g_surface);
    }
    if (g_context) {
        env->DeleteGlobalRef(g_context);
    }
    g_surface = env->NewGlobalRef(surface);
    g_context = env->NewGlobalRef(context);

    // writes stdout to logcat
    static std::once_flag piped;
    std::call_once(piped, []() { pipestd(); });

    // get args
    jni::StringRef urlString(env, url);
    jni::StringRef filterString(env, filter);
    jni::StringRef xmlString(env, xml);
    std::vector<const char*> args;
    args.push_back("TestRunner");
    if (!urlString.get().empty()) {
        args.push_back(urlString.get().c_str());
    }
    if (!filterString.get().empty()) {
        args.push_back(filterString.get().c_str());
    }
    if (!xmlString.get().empty()) {
        args.push_back(xmlString.get().c_str());
    }

    // run tests
    if (!twitch::IntegrationTest::initialize(static_cast<int>(args.size()), const_cast<char**>( &args[0]))) {
        return -1;
    }

    __android_log_write(ANDROID_LOG_INFO, "Twitch", "BEGIN_TEST");
    int testResult = twitch::IntegrationTest::run();
    __android_log_write(ANDROID_LOG_INFO, "Twitch", "END_TEST");
    return testResult;
}
}

namespace twitch {
namespace test {
// Implement createPlayer (see test/common/testenv/android/testenv.cpp)
std::shared_ptr<MediaPlayer> TestEnvironment::createPlayer(Player::Listener& listener)
{
    auto player = std::make_shared<MediaPlayer>(listener, createPlatform());
    player->setSurface(g_surface);
    return player;
}

std::shared_ptr<Platform> TestEnvironment::createPlatform()
{
    std::shared_ptr<twitch::AndroidPlatform> platform;
    if (jni::getVM()) {
        jni::AttachThread attachThread(jni::getVM());
        JNIEnv* env = attachThread.getEnv();
        if (env) {
            platform = twitch::AndroidPlatform::create(env, g_context);
        }
    }
    return platform;
}
}
}
