#import <UIKit/UIKit.h>
#include <cstdlib>
#include <future>
#include <gtest/gtest.h>
#include <iostream>
#include <unistd.h>
#include "IntegrationTest.hpp"
#include "platforms/ios/TTVPlayer.h"
#include "platforms/ios/TTVPlayerLayer.h"
#include "platforms/ios/ApplePlatform.h"
#include "signalhandler.hpp"
#include "testenv.hpp"

/* External dependency */
namespace twitch {
    namespace test {
        void setTestLayer(CALayer* layer);
    }
}

/* Logger */
class TestLogger {
public:
    TestLogger()
    : handle(nullptr)
    {}
    ~TestLogger() {
        close();
    }
    void open(const std::string& logfile) {
        if (!handle) handle = fopen(logfile.c_str(), "w");
        if (!handle) {
            NSLog(@"WARNING: Unable to open logfile: %s (%s)", logfile.c_str(), strerror(errno));
        } else {
            NSLog(@"Opened logfile '%s' for writing", logfile.c_str());
        }
    }
    void log(const char* fmt, ...) {
        va_list args;
        va_start(args, fmt);
        char buffer[2048];
        buffer[sizeof(buffer)-1] = '\0';
        vsnprintf(buffer, sizeof(buffer), fmt, args);
        if (handle) fprintf(handle, "%s", buffer);
        va_end(args);
        if (handle) fflush(handle);
#if !TARGET_OS_SIMULATOR
        NSLog(@"%s", buffer);
#endif
    }
    void close() {
        if (handle) {
            NSLog(@"Closing logfile");
            fclose(handle);
            handle = nullptr;
        }
    }
private:
    FILE* handle;
};
static TestLogger testLogger;

/* Argument store */
struct ArgStore {
    ArgStore()
        : argc(0)
        , argv(nullptr)
        , xcTestEnabled(true)
        , logfile()
        , xmlfile()
    {}
    
    /* Cache arguments and parse */
    void set(int _argc, char** _argv) {
        if ( argc != _argc || argv != _argv ) {
            argc = _argc;
            argv = _argv;
            if ( parseArguments(argc, argv) ) {
                assert(logfile.size() > 0);
                testLogger.open(logfile.c_str());
                NSLog(@"XCTest enabled: %s", xcTestEnabled ? "Yes" : "No");
                NSLog(@"Writing logfile to: %s", logfile.c_str());
                int i = 0;
                while (i < argc) {
                    if ( argv[i] != nullptr ) {
                        NSLog(@"- Arg[%d]: %s", i, argv[i]);
                    }
                    ++i;
                }
            }
        }
    }
    
    /* Parse arguments */
    bool parseArguments(int argc, char** argv) {
        bool valid = false;
        for (int i = 0; i < argc; i++) {
            if (argv[i] != nullptr) {
                std::string arg(argv[i]);
                size_t pos = arg.find("--logfile=");
                if (pos != std::string::npos) {
                    logfile = arg.substr(sizeof("--logfile=") - 1);
                    valid = true;
                    continue;
                }
                pos = arg.find("--noxctest");
                if (pos != std::string::npos) {
                    xcTestEnabled = false;
                    continue;
                }
                pos = arg.find("--gtest_output=xml:");
                if (pos != std::string::npos) {
                    xmlfile = arg.substr(sizeof("--gtest_output=xml:") - 1);
                    continue;
                }
            }
        }
        return valid;
    }
    int argc;
    char** argv;
    bool xcTestEnabled;
    std::string logfile;
    std::string xmlfile;
};
static ArgStore argStore;

/* Returns whether XCTest is enabled */
static bool isXCTestEnabled() {
    return argStore.xcTestEnabled;
}

/** Print a file to stdout */
void printFile(const std::string& path) {
    if (path.size() > 0) {
        FILE* handle = fopen(path.c_str(), "rb");
        if ( handle ) {
            fseek(handle, 0, SEEK_END);
            size_t size = ftell(handle);
            rewind(handle);
            char* buffer = new char[size+1];
            buffer[size] = '\0';
            fread(buffer, 1, size, handle);
            fclose(handle);
            NSLog(@"[[PRINT_START]]%s[[PRINT_END]]", buffer);
            delete[](buffer);
        }
    }
}

/* Call into framework */
static int frameworkCall(int argc, char **argv) {
    int result = 1;
    try {
        if (twitch::IntegrationTest::initialize(argc, argv)) {
            result = twitch::IntegrationTest::run();
        }
    } catch (std::exception e) {
        testLogger.log("Caught exception: %s\n", e.what());
    } catch (...) {
        testLogger.log("Unhandled exception caught during testing\n");
    }
    return result;
}

/* Execute testing framework */
int runIntegrationTestFramework(int argc, char** argv) {
    argStore.set(argc, argv);
    for (int i = 0; i < argc; ++i) {
        testLogger.log("argv[%d]: %s", i, argv[i]);
    }
    __block int result = 0;
    if (isXCTestEnabled()) {
        result = frameworkCall(argc, argv);
#if !TARGET_OS_SIMULATOR
        printFile(argStore.xmlfile);
#endif
    } else {
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            result = frameworkCall(argc, argv);
        });
    }
    return result;
}

/* View Controller */
@interface ViewController : UIViewController
- (void)viewDidLoad;
@end
@implementation ViewController
- (void)viewDidLoad {
    CALayer* playerLayer;
#ifdef ENABLE_LOOPBACK
    playerLayer = (AVPlayerLayer*)[[AVPlayerLayer alloc] init];
#else
    playerLayer = (TTVPlayerLayer*)[[TTVPlayerLayer alloc] init];
#endif
    // Configure player instance
    twitch::test::setTestLayer(playerLayer);
    playerLayer.frame = self.view.bounds;
    TTVPlayer* player = [[TTVPlayer alloc] initWithLayer: playerLayer];
    [self.view.layer addSublayer: playerLayer];
    player.delegate = nil;
    
    // TestDriver (XCTest) is calling this externally
    if ( !isXCTestEnabled() ) {
        runIntegrationTestFramework(argStore.argc, argStore.argv);
    }
}
@end

/* App Delegate */
@interface AppDelegate : UIResponder <UIApplicationDelegate>
@property (strong, nonatomic) UIWindow* window;
@end
@implementation AppDelegate
- (BOOL)application:(UIApplication*)application didFinishLaunchingWithOptions:(NSDictionary*)launchOptions {
    // Override point for customization after application launch.
    (void)application;
    (void)launchOptions;
    return YES;
}
@end

/* Entry Point */
int main(int argc, char* argv[]) {
    argStore.set(argc, argv);
    @autoreleasepool {
        return UIApplicationMain(argc, argv, nil, NSStringFromClass([AppDelegate class]));
    }
}
