#pragma once
#include "player_state_timer.hpp"
#include "player_test_base.hpp"
#include "player_test_timeout.hpp"
#include "time_util.hpp"
#include "timed_test_runner.hpp"
#include <algorithm>
#include <cmath>
#include <functional>
#include <gtest/gtest.h>
#include <iostream>
#include <thread>
#include <utility>
#include <vector>
#include "IntegrationTest.hpp"
#include "debug/debug.hpp"
#include "player/AsyncMediaPlayer.hpp"
#include "player/MediaPlayer.hpp"
#include <gtest/gtest.h>
#include "test/common/util/gtestenv.hpp"

// Note:    ALL Testing on Android disabled, because of numerous flaky tests
//          https://jira.twitch.com/browse/VP-5921
//
#if !PLAYERCORE_OS_ANDROID

/**
 * AsyncPlayerTest validates anything about the deferred facade/proxy player
 */

using namespace twitch;
using namespace testing;


namespace twitch {
    struct MediaSample;
}

/**
 * Base class for MediaPlayer tests that will run on a given parameterized
 * set of test stream URL's
 */
class AsyncPlayerTestBaseWithListener : public ::testing::Test, public twitch::Player::Listener {
public:
    AsyncPlayerTestBaseWithListener()
        : m_environment(IntegrationTest::environment)
    {
        assert(m_environment);
    }

    ~AsyncPlayerTestBaseWithListener() override = default;

    void SetUp() override
    {
        TraceLog::get().setLevel(IntegrationTest::environment->options().getLogLevel());

        auto platform = GTestEnvironment::current->createPlatform();
        m_player = std::make_shared<AsyncMediaPlayer>(*this, std::static_pointer_cast<twitch::NativePlatform>(platform));

        m_stateMonitor = std::unique_ptr<PlayerStateChangeMonitor>(new PlayerStateChangeMonitor(m_player->getState()));
        m_callbackMonitor = std::unique_ptr<PlayerCallbackMonitor>(new PlayerCallbackMonitor());

        bool isLive = IntegrationTest::environment->testData().isLive;
        m_stateMachine = std::unique_ptr<PlayerStateMachine>(new PlayerStateMachine(m_player->getState(), isLive));

        m_url = IntegrationTest::environment->testData().url;
    }

    void TearDown() override
    {
        m_callbackMonitor.reset();
        m_stateMonitor.reset();
    }

protected:
    struct ListenerExecution
    {
        ListenerExecution()
        {
            clear();
        }

        void clear()
        {
            onDurationChanged = false;
            onMetadata = false;
            onError = false;
            onPositionChanged = false;
            onQualityChanged = false;
            onSeekCompleted = false;
            onStateChanged = false;
            onAnalyticsEvent = false;
        }

        bool onDurationChanged;
        bool onMetadata;
        bool onError;
        bool onPositionChanged;
        bool onQualityChanged;
        bool onSeekCompleted;
        bool onStateChanged;
        bool onAnalyticsEvent;
    };

    ListenerExecution m_listenerExecution;

    // Player::Listener overrides
    void onDurationChanged(twitch::MediaTime) override
    {
        m_listenerExecution.onDurationChanged = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onError(const Error&) override
    {
        m_listenerExecution.onError = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onMetadata(const std::string&, const std::vector<uint8_t>&) override
    {
        m_listenerExecution.onMetadata = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onPositionChanged(MediaTime) override
    {
        m_listenerExecution.onPositionChanged = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onQualityChanged(const Quality&) override
    {
        m_listenerExecution.onQualityChanged = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onRebuffering() override 
    {
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onRecoverableError(const twitch::Error&) override 
    {
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onSeekCompleted(twitch::MediaTime time) override
    {
        m_listenerExecution.onSeekCompleted = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);

        if (m_callbackMonitor) {
            m_callbackMonitor->onSeekCompleted(time);
        }
    }

    void onSessionData(const std::map<std::string, std::string>& properties) override
    {
        (void)properties;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

    void onStateChanged(twitch::Player::State state) override
    {
        m_listenerExecution.onStateChanged = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);

        if (m_stateMonitor) {
            m_stateMonitor->onStateChanged(state);
        }
    }

    void onAnalyticsEvent(const std::string&, const std::string&) override
    {
        m_listenerExecution.onAnalyticsEvent = true;
        m_player->thisThreadShouldRunInCommandThread(__FUNCTION__);
    }

protected:
    virtual void onRender(const twitch::MediaSample& sample, int width, int height)
    {
        (void)sample;
        (void)width;
        (void)height;
    }

    const twitch::IntegrationTestEnvironment::TestInputData& testInput() const { return m_environment->testData(); }

    const twitch::IntegrationTestEnvironment* m_environment;

    // MediaPlayer parameters
    std::shared_ptr<twitch::Log> m_log;

    // Deferred MediaPlayer
    std::shared_ptr<twitch::AsyncMediaPlayer> m_player;

    // MediaPlayer::load() parameter
    std::string m_url;

    // Every integration test is going to need to know about the player state
    std::unique_ptr<PlayerStateChangeMonitor> m_stateMonitor;
    std::unique_ptr<PlayerCallbackMonitor> m_callbackMonitor;
    std::unique_ptr<PlayerStateMachine> m_stateMachine;
};

class AsyncPlayerTestBase : public AsyncPlayerTestBaseWithListener {
public:
    ~AsyncPlayerTestBase() override = default;

    void SetUp() override
    {
        AsyncPlayerTestBaseWithListener::SetUp();
        m_stateTimer = std::unique_ptr<PlayerStateTimer>(new PlayerStateTimer(m_player->getState()));
    }

    void TearDown() override
    {
        m_stateTimer.reset();
        AsyncPlayerTestBaseWithListener::TearDown();
    }

    void onStateChanged(twitch::Player::State state) override
    {
        AsyncPlayerTestBaseWithListener::onStateChanged(state);
        if (m_stateTimer) {
            m_stateTimer->onStateChanged(state);
        }
    }

    testing::AssertionResult testCompletedExecution(TimedTestRunner::time_t testTime)
    {
        (void)testTime;

        return ::testing::AssertionSuccess();
    }

    testing::AssertionResult testCompletedExecutionWithQualityCallback(TimedTestRunner::time_t testTime)
    {
        if (!m_onQualityChanged)  {
            return ::testing::AssertionFailure() << " OnQualityChanged callback not called" << TimeUtil::toString<std::chrono::milliseconds>(m_testStartPosition.microseconds());
        }

        return testCompletedExecution(testTime);
    }

    
protected:
    std::unique_ptr<PlayerStateTimer> m_stateTimer;
    bool m_onQualityChanged = false;
    twitch::MediaTime m_testStartPosition = twitch::MediaTime::zero();
};

class AsyncPlayerTest : public AsyncPlayerTestBase, public testing::WithParamInterface<std::string> {
public:
    ~AsyncPlayerTest() override = default;

    void SetUp() override
    {
        AsyncPlayerTestBase::SetUp();
    }
};

class AsyncPlayer : public AsyncPlayerTestBase {
public:
    ~AsyncPlayer() override = default;

    void SetUp() override
    {
        AsyncPlayerTestBase::SetUp();
        m_url = "https://www.twitch.tv/videos/v205550776";
        
        m_player->enableDebuggingCommand(true);
    }
};

TEST_F(AsyncPlayer, simpleLoad)
{
    m_url = "https://www.twitch.tv/videos/164446609";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(AsyncPlayer, singleThreadExecution)
{
    m_url = "https://www.twitch.tv/videos/v205550776";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";
    
    std::vector<std::thread> workers;
    for (int i = 0; i < 4; i++) {
        workers.push_back(std::thread([this]()
        {
            for (int tryIndex = 0; tryIndex < 16; tryIndex++)
            {
                if ((rand() & 1) != 0) {
                    m_player->pause();
                } else {
                    m_player->play();
                }

                std::this_thread::sleep_for(std::chrono::milliseconds((rand() & 63) + 1));
            }
        }));
    }

    std::for_each(workers.begin(), workers.end(), [](std::thread & t) {
        t.join();
    });

    // Added 3 seconds to execute all commands.
    std::this_thread::sleep_for(std::chrono::seconds(3));

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(AsyncPlayer, ValidateProtectedFunctions)
{
    m_player->disableCommandExecution(true);

    size_t currentCommandId;
    Quality quality;

    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->load(m_url);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->load(m_url, "");
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->play();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->pause();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->seekTo(MediaTime::zero());
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->isSeekable();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getDuration();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getPosition();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getBufferedPosition();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getState();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getVersion();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->isLooping();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setLooping(true);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->isMuted();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setMuted(true);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getVolume();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setVolume(true);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getQuality();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setQuality(quality);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setQuality(quality, true);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getQualities();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getAutoSwitchQuality();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setAutoSwitchQuality(true);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getAverageBitrate();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getBandwidthEstimate();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getPlaybackRate();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setPlaybackRate(1.0f);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getPath();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setClientId("");
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setDeviceId("");
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setAuthToken("");
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getLiveLatency();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setAutoInitialBitrate(1);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setAutoMaxBitrate(1);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setAutoMaxVideoSize(0, 0);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setLiveMaxLatency(MediaTime::zero());
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setLiveLowLatencyEnabled(true);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setMinBuffer(MediaTime::zero());
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setMaxBuffer(MediaTime::zero());
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->getStatistics();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->regenerateAnalyticsPlaySession();
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }
    {
        currentCommandId = m_player->getCurrentCommandId();
        m_player->setKeepAnalyticsPlaySession(true);
        ASSERT_TRUE(currentCommandId != m_player->getCurrentCommandId()) << "Function is not protected: " << m_player->getQueueSize();
    }

    m_player->disableCommandExecution(false);
}

#if PLAYERCORE_OS_IOS
// https://jira.twitch.com/browse/VP-5921
TEST_F(AsyncPlayer, DISABLED_testListenerFunctions)
#else
TEST_F(AsyncPlayer, testListenerFunctions)
#endif
{
    m_url = "https://www.twitch.tv/videos/v205550776";

    m_listenerExecution.clear();

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();

        twitch::MediaTime seekPos(1.0f);
        m_player->seekTo(seekPos);
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    // Added 3 seconds to execute all commands.
    std::this_thread::sleep_for(std::chrono::seconds(3));

    ASSERT_TRUE(m_listenerExecution.onDurationChanged) << "OnDurationChanged listener is not executed";
    ASSERT_TRUE(m_listenerExecution.onMetadata) << "OnMetadataChanged listener is not executed";
    ASSERT_TRUE(m_listenerExecution.onPositionChanged) << "OnPositionChanged listener is not executed";
    ASSERT_TRUE(m_listenerExecution.onQualityChanged) << "OnQualityChanged listener is not executed";

    ASSERT_TRUE(m_listenerExecution.onSeekCompleted) << "onSeekCompleted listener is not executed";
    ASSERT_TRUE(m_listenerExecution.onStateChanged) << "onStateChanged listener is not executed";
    ASSERT_TRUE(m_listenerExecution.onAnalyticsEvent) << "onAnalyticsEvent listener is not executed";
    
    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(AsyncPlayer, TestListenerError)
{
    m_url = "127.0.0.1";

    m_listenerExecution.clear();

    m_player->load(m_url);

    // Added 3 seconds to execute all commands.
    std::this_thread::sleep_for(std::chrono::seconds(3));

    ASSERT_TRUE(m_listenerExecution.onError) << "OnError listener is not executed";

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

class AsyncPlayerReentrance : public AsyncPlayer {
public:
    ~AsyncPlayerReentrance() override = default;

    void onStateChanged(twitch::Player::State state) override
    {
        if (m_stateMonitor) {
            m_stateMonitor->onStateChanged(state);
        }

        static int toggleLimit = 12;
        if (--toggleLimit > 0) {
            if (state == Player::State::Playing) {
                m_player->pause();
            }
            else {
                m_player->play();
            }
        }
    }
};

TEST_F(AsyncPlayerReentrance, PlayPauseNonStop)
{
    m_url = "https://www.twitch.tv/videos/164446609";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    // Added 3 seconds to execute all commands.
    std::this_thread::sleep_for(std::chrono::seconds(3));
}

class AsyncPlayerGetDuration : public AsyncPlayer {
public:
    ~AsyncPlayerGetDuration() override = default;

    void onDurationChanged(twitch::MediaTime duration) override
    {       
        m_duration = duration;
    }

    twitch::MediaTime m_duration;
};

TEST_F(AsyncPlayerGetDuration, GetDuration)
{
    m_url = "https://www.twitch.tv/videos/164446609";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    std::this_thread::sleep_for(std::chrono::seconds(1));

    twitch::MediaTime duration = m_player->getDuration();
    
    ASSERT_TRUE(duration == m_duration) << m_duration.seconds() << " " << duration.seconds();
    ASSERT_TRUE(duration != twitch::MediaTime::zero()) << m_duration.seconds() << " " << duration.seconds();
}

class AsyncPlayerGetPosition : public AsyncPlayer {
public:
    ~AsyncPlayerGetPosition() override = default;

    void onPositionChanged(MediaTime position) override
    {
        MediaTime positionFromGet = m_player->getPosition();
        ASSERT_TRUE(position == positionFromGet) << position.seconds() << " " << positionFromGet.seconds();
    }
};

TEST_F(AsyncPlayerGetPosition, getPosition)
{
    m_url = "https://www.twitch.tv/videos/164446609";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    std::this_thread::sleep_for(std::chrono::seconds(1));
}

TEST_F(AsyncPlayer, GetSetFunctions)
{
    m_url = "https://www.twitch.tv/videos/164446609";

    const float playbackRate = 1.10f;
    const float volume = 0.5f;

    m_player->setPlaybackRate(playbackRate);
    m_player->setLooping(true);
    m_player->setMuted(true);    
    m_player->setVolume(volume);

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    ASSERT_TRUE(playbackRate == m_player->getPlaybackRate()) << playbackRate << " " << m_player->getPlaybackRate();
    ASSERT_TRUE(twitch::Player::State::Playing == m_player->getState()) << static_cast<int>(m_player->getState());
    ASSERT_TRUE(m_player->isMuted()) << m_player->isMuted();
    ASSERT_TRUE(m_player->isLooping()) << m_player->isLooping();
    ASSERT_TRUE(volume == m_player->getVolume()) << volume << " " << m_player->getVolume();

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

class AsyncPlayerIsSeekableTrue : public AsyncPlayer {
public:
    ~AsyncPlayerIsSeekableTrue() override = default;

    void onPositionChanged(MediaTime) override
    {        
        ASSERT_TRUE(m_player->isSeekable()) << m_player->isSeekable();
    }
};

TEST_F(AsyncPlayerIsSeekableTrue, isSeekable)
{
    m_url = "https://www.twitch.tv/videos/164446609";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

class AsyncPlayerIsSeekableFalse : public AsyncPlayer {
public:
    ~AsyncPlayerIsSeekableFalse() override = default;

    void onPositionChanged(MediaTime) override
    {
        ASSERT_TRUE(!m_player->isSeekable()) << m_player->isSeekable();
    }
};

TEST_F(AsyncPlayerIsSeekableFalse, isSeekable)
{
    m_url = "https://www.twitch.tv/monstercat";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

class AsyncPlayerGetPath : public AsyncPlayer {
public:
    ~AsyncPlayerGetPath() override = default;

    void onStateChanged(twitch::Player::State state) override
    {
        AsyncPlayer::onStateChanged(state);

        if (state == twitch::Player::State::Playing) {
            ASSERT_TRUE(m_url == m_player->getPath()) << m_url << " " << m_player->getPath();
        }
    }
};

TEST_F(AsyncPlayerGetPath, getPath)
{
    m_url = "https://www.twitch.tv/monstercat";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Idle, [this]() {
        m_player->pause();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Pause state for test setup";
    
    m_url = "https://www.twitch.tv/videos/164446609";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

class AsyncPlayerGetQuality : public AsyncPlayer {
public:
    ~AsyncPlayerGetQuality() override = default;

    void onQualityChanged(const Quality& quality) override
    {
        ASSERT_TRUE(quality == m_player->getQuality()) << quality.name << " " << m_player->getQuality().name;
        m_onQualityChanged = true;
    }
};

TEST_F(AsyncPlayerGetQuality, getQualities)
{
    m_url = "https://www.twitch.tv/videos/v205550776";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    size_t qualityCount = m_player->getQualities().size();
    ASSERT_TRUE(qualityCount != 0) << qualityCount;

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecutionWithQualityCallback, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}


#if PLAYERCORE_OS_IOS
// https://jira.twitch.com/browse/VP-5921
TEST_F(AsyncPlayer, DISABLED_getQualityFromCallback)
#else
TEST_F(AsyncPlayer, getQualityFromCallback)
#endif
{
    m_url = "https://www.twitch.tv/monstercat";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";    

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecutionWithQualityCallback, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(AsyncPlayerGetQuality, SetAndGetQuality)
{
    m_url = "https://www.twitch.tv/monstercat";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";    

    size_t qualityCount = m_player->getQualities().size();
    ASSERT_TRUE(qualityCount != 0) << qualityCount;

    if (qualityCount > 0) {
        m_player->setQuality(m_player->getQualities()[qualityCount - 1]);
    }

    // Added 3 seconds to apply the quality.
    std::this_thread::sleep_for(std::chrono::seconds(3));

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecutionWithQualityCallback, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(AsyncPlayer, getBandwidthEstimate)
{
    m_player->thisMethodShouldRunInMainThread(__FUNCTION__);

    m_url = "https://www.twitch.tv/videos/164446609";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    for (int tryIndex = 0; tryIndex < 16; tryIndex++)
    {
        int bandwidthEstimate = m_player->getBandwidthEstimate();
        ASSERT_TRUE(bandwidthEstimate == 0) << "bandwidthEstimate: " << bandwidthEstimate << ". You update this function (getBandwidthEstimate), fix this test by validate the result";
        std::this_thread::sleep_for(std::chrono::milliseconds((rand() & 63) + 1));
    }

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

#if PLAYERCORE_OS_IOS
// https://jira.twitch.com/browse/VP-5921
TEST_F(AsyncPlayer, DISABLED_simulateVideoError)
#else
TEST_F(AsyncPlayer, simulateVideoError)
#endif
{
    m_url = "https://www.twitch.tv/videos/v205550776";

    m_listenerExecution.clear();

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    twitch::debug::simulateVideoError(m_player);

    // Added 3 seconds to execute all commands.
    std::this_thread::sleep_for(std::chrono::seconds(3));

    ASSERT_TRUE(m_listenerExecution.onError) << "OnError listener is not executed";

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&AsyncPlayerTest::testCompletedExecution, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

#endif // PLAYERCORE_OS_ANDROID

