#pragma once
#include "player_test_base.hpp"
#include "player_test_timeout.hpp"
#include "analytics/AnalyticsTracker.hpp"
#include "player_listener.hpp"
#include "playercore/MediaTime.hpp"
#include <functional>
#include <json11.hpp>
#include "test/common/util/event_listener.hpp"
#include "test/common/util/event_receiver.hpp"

/**
 * Validate Player::onAnalyticsEvent() events
 */
using namespace twitch;
using namespace twitch::test;
class PlayerAnalyticsEventTest : public PlayerTestBase {
protected:
    using OnTrackingHandlerType = EventHandler<const std::string&, const std::string&>;
    using OnTrackingReceivedType = typename OnTrackingHandlerType::EventReceivedType;

    PlayerEventListener m_listener;
    std::string m_eventName;
    std::thread m_thread;

    void SetUp() override
    {
        PlayerTestBase::SetUp();
    }

    void TearDown() override
    {
        if (m_thread.joinable()) {
            m_thread.join();
        }
        PlayerTestBase::TearDown();
    }

    void onAnalyticsEvent(const std::string& name, const std::string& properties) override
    {
        if (m_eventName == name) {
            m_listener.onAnalyticsEvent(name, properties);
        }
    }

    void validateLiveProperty(const json11::Json::object& properties)
    {
        const auto& itr = properties.find("live");
        ASSERT_NE(itr, properties.end());

        const auto& live = itr->second;
        EXPECT_FALSE(live.is_null());
        EXPECT_TRUE(live.is_bool());

        EXPECT_EQ(testInput().isLive, live.bool_value());
    }
};

class VideoPlayEventTest : public PlayerAnalyticsEventTest {
protected:
    MediaTime m_eventReceived = MediaTime::zero();

    void SetUp() override
    {
        PlayerAnalyticsEventTest::SetUp();
        m_eventName = "video-play";
    }

    void onAnalyticsEvent(const std::string& name, const std::string& properties) override
    {
        if (name == m_eventName) {
            m_eventReceived = MediaTime::now();
            PlayerAnalyticsEventTest::onAnalyticsEvent(name, properties);
        }
    }
};

TEST_F(VideoPlayEventTest, VideoPlayOnPlay)
{
    const auto timeout = PlayerTestTimeout::load + PlayerTestTimeout::start;
    MediaTime testStarted = MediaTime::zero();
    OnTrackingReceivedType trackingEvent;

    ASSERT_TRUE(EventReceiver::waitFor(&m_listener.onTrackingEvent, trackingEvent, timeout, [&]() {
        m_thread = std::thread([&]() {
            testStarted = MediaTime::now();
            m_player->load(m_url);
            m_player->play();
        });
    })) << "Test timed out waiting for onTracking event";

    const std::string& name = std::get<0>(trackingEvent);
    const std::string& properties = std::get<1>(trackingEvent);

    ASSERT_FALSE(name.empty());
    ASSERT_FALSE(properties.empty());

    EXPECT_EQ("video-play", name);

    std::string error;
    const auto& propertiesJson = json11::Json::parse(properties, error);
    EXPECT_FALSE(propertiesJson.is_null()) << "Error parsing JSON: " << error;
    ASSERT_TRUE(propertiesJson.is_object()) << "Expected JSON to be object";

    // Valid time to first frame
    ASSERT_GT(testStarted, MediaTime::zero());
    ASSERT_GT(m_eventReceived, MediaTime::zero());

    auto eventTime = m_eventReceived - testStarted;
    ASSERT_GT(eventTime, MediaTime::zero());

    const auto& timeSinceLoadStartProperty = propertiesJson["time_since_load_start"];
    EXPECT_FALSE(timeSinceLoadStartProperty.is_null());

    auto timeSinceLoadStartMsec = timeSinceLoadStartProperty.int_value();
    EXPECT_GT(timeSinceLoadStartMsec, 0);

    const MediaTime episilon(std::chrono::seconds(1));
    EXPECT_GT(timeSinceLoadStartMsec, static_cast<int>((eventTime - episilon).milliseconds().count()));

    validateLiveProperty(propertiesJson.object_items());
}
