#pragma once
#include "event_receiver.hpp"
#include "player_test_base.hpp"
#include "player_test_timeout.hpp"
#include <condition_variable>
#include <json11.hpp>
#include <mutex>

using namespace twitch::test;
/**
 * Parameterized tests for MediaPlayer's metadata
 */
class PlayerMetadataTest : public PlayerTestBase {
protected:
    using OnMetadataHandlerType = EventHandler<const std::string&, const std::vector<uint8_t>&>;
    using OnMetadataReceivedType = typename OnMetadataHandlerType::EventReceivedType;

    void SetUp() override
    {
        PlayerTestBase::SetUp();
    }

    void TearDown() override
    {
        PlayerTestBase::TearDown();
    }

    void onMetadata(const std::string& name, const std::vector<uint8_t>& data) override
    {
        m_listener.onMetadata(name, data);
    }

    bool hasID3Id(const std::string& id, const json11::Json::array& id3Tags)
    {
        // Convenience function to determine
        for (const auto& id3Tag : id3Tags) {
            if (!id3Tag.is_object()) {
                TRACE_ERROR("Unexpected ID3 format: %s", id3Tag.string_value().c_str());
                return false;
            }

            const auto& id3TagId = id3Tag["id"];
            if (!id3TagId.is_string()) {
                TRACE_ERROR("Unexpected ID3 format: ID3 'id' is not a string");
                return false;
            }

            if (id == id3TagId.string_value()) {
                return true;
            }
        }

        return false;
    }

    void validate(const std::string& type, const std::vector<uint8_t>& data)
    {
        ASSERT_FALSE(type.empty());
        ASSERT_FALSE(data.empty());

        if (type == twitch::MediaType::Text_Json.name) {

            std::string error;
            std::string text(data.data(), data.data() + data.size());
            const auto json = json11::Json::parse(text, error);
            EXPECT_FALSE(json.is_null()) << "Error parsing JSON: " << error;

            ASSERT_TRUE(json.is_object()) << "Expected JSON to be object";

            if (!json["ID3"].array_items().empty()) {
                const auto& id3 = json["ID3"];
                const auto& id3Tags = id3.array_items();
                for (const auto& tag : m_tagsToValidate) {
                    EXPECT_TRUE(hasID3Id(tag, id3Tags)) << "Could not find ID3 tag: " << tag;
                }
            }
        }
    }

    const std::vector<std::string> m_tagsToValidate{ "TDEN", "TDTG" };
    PlayerEventListener m_listener;
};

TEST_F(PlayerMetadataTest, ValidateID3)
{
    OnMetadataReceivedType received;
    auto timeout = PlayerTestTimeout::load + PlayerTestTimeout::start;

    ASSERT_TRUE(EventReceiver::waitFor(&m_listener.onMetadataEvent, received, timeout, [this]() {
        m_player->load(m_url);
        m_player->play();
    })) << "Test timed out waiting for onMetadata event";

    validate(std::get<0>(received), std::get<1>(received));
}
