#pragma once
#include "IntegrationTest.hpp"
#include "playercore/Quality.hpp"
#include "player_test_base.hpp"
#include "player_test_timeout.hpp"
#include "quality_monitor.hpp"
#include "TestStreamConfig.hpp"
#include "test_util.hpp"
#include <algorithm>

using namespace twitch::test;
/**
 * Parameterized tests for MediaPlayer quality changes
 */
class PlayerQualityTest : public PlayerTestBase {
protected:
    static void SetUpTestCase()
    {
        const auto& url = twitch::IntegrationTest::environment->testData().url;
        auto testStream = std::unique_ptr<TestStreamConfig>(new TestStreamConfig(url));
        testStream->init() << "Could not initialize TestStreamConfig for " << url;
        TestStream = std::move(testStream);
    }

    static void TearDownTestCase()
    {
        TestStream.reset();
    }

protected:
    void SetUp() override
    {
        PlayerTestBase::SetUp();

        ASSERT_TRUE(TestStream->isInitialized) << "Test stream configuration could not initialize for: " << m_url;
        ASSERT_FALSE(m_url.empty());
        ASSERT_GT(TestStream->expectedQualities.size(), 1U) << "Need multiple quality variants for Quality tests";

        m_player->setAutoSwitchQuality(false);
        m_qualityControlMonitor = std::unique_ptr<QualityMonitor>(new QualityMonitor(m_player->getQuality()));
    }

    void TearDown() override
    {
        PlayerTestBase::TearDown();
        m_qualityControlMonitor.reset();
    }

protected:
    void onQualityChanged(const twitch::Quality& quality) override
    {
        if (m_qualityControlMonitor) {
            m_qualityControlMonitor->onQualityChanged(quality);
        }
    }

    twitch::Quality getQuality(const std::string& name) const
    {
        const auto& qualities = m_player->getQualities();
        const auto itr = std::find_if(qualities.begin(), qualities.end(), [&](const twitch::Quality& quality) {
            return quality.name == name;
        });

        return itr == qualities.end() ? twitch::Quality() : *itr;
    }

    static std::unique_ptr<const TestStreamConfig> TestStream;

    std::unique_ptr<QualityMonitor> m_qualityControlMonitor;
};

std::unique_ptr<const TestStreamConfig> PlayerQualityTest::TestStream = nullptr;

#if defined(__ANDROID__)
// CVP-2474
TEST_F(PlayerQualityTest, DISABLED_GetQualities)
#else
TEST_F(PlayerQualityTest, GetQualities)
#endif
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Could not load stream";

    const auto& qualities = m_player->getQualities();
    const size_t numQualities = qualities.size();
    ASSERT_EQ(TestStream->expectedQualities.size(), numQualities);

    const twitch::Quality* lastQuality = nullptr;

    for (size_t i = 0; i < numQualities; i++) {
        const auto& currentQuality = qualities[i];
        const auto& expectedQuality = TestStream->expectedQualities[i];
        EXPECT_EQ(expectedQuality.name, currentQuality.name);
        EXPECT_EQ(expectedQuality.height, currentQuality.height);
        EXPECT_EQ(expectedQuality.width, currentQuality.width);

        if (lastQuality) {
            EXPECT_GE(lastQuality->bitrate, currentQuality.bitrate) << "Bitrate not in descending order";

            EXPECT_GE(lastQuality->height, currentQuality.height) << "Quality height not in descending order";
            EXPECT_GE(lastQuality->width, currentQuality.width) << "Quality width not in descending order";
        }

        if (!currentQuality.isDefault) {
            lastQuality = &currentQuality;
        }
    }
}

#if defined(__ANDROID__)
// CVP-2473
TEST_F(PlayerQualityTest, DISABLED_StartsAtDefaultQuality)
#else
TEST_F(PlayerQualityTest, StartsAtDefaultQuality)
#endif
{
    const auto& defaultQuality = TestStream->defaultQuality;

    bool waitedFor = m_qualityControlMonitor->waitFor(defaultQuality.name, [&]() {
        // NOTE: onQualityChange fires AFTER Playing state, so nest these test expectations so that the quality change
        ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
            m_player->load(m_url);
            m_player->play();
        },
            PlayerTestTimeout::load + PlayerTestTimeout::start))
            << "Could not load stream";
    },
        PlayerTestTimeout::qualityChange);

    EXPECT_TRUE(waitedFor) << "Player timed out trying to switch to quality: " << defaultQuality.name;

    const auto& actualQuality = m_player->getQuality();
    EXPECT_EQ(defaultQuality.name, actualQuality.name);
    EXPECT_EQ(defaultQuality.height, actualQuality.height);
    EXPECT_EQ(defaultQuality.width, actualQuality.width);
}

#if defined(__ANDROID__)
// CVP-2475
TEST_F(PlayerQualityTest, DISABLED_SwitchToLowest)
#else
TEST_F(PlayerQualityTest, SwitchToLowest)
#endif
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Could not load stream";

    const size_t numQualities = TestStream->expectedQualities.size();
    const auto& expectedQuality = TestStream->expectedQualities[numQualities - 1];
    ASSERT_TRUE(expectedQuality.isValid()) << "Could not find expected quality: " << expectedQuality.name;

    const auto targetQuality = getQuality(expectedQuality.name);
    ASSERT_FALSE(targetQuality.name.empty()) << "Could not find target quality: " << expectedQuality.name;

    ASSERT_TRUE(m_qualityControlMonitor->waitFor(expectedQuality.name, [&]() {
        m_player->setQuality(targetQuality);
    },
        PlayerTestTimeout::qualityChange))
        << "Player timed out trying to switch to quality: " << expectedQuality.name;

    const auto& actual = m_player->getQuality();
    EXPECT_EQ(expectedQuality.height, actual.height);
    EXPECT_EQ(expectedQuality.width, actual.width);
}

#if defined(__ANDROID__)
TEST_F(PlayerQualityTest, DISABLED_SwitchToMiddle) // CVP-2090
#else
TEST_F(PlayerQualityTest, SwitchToMiddle)
#endif
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Could not load stream";

    const size_t numQualities = TestStream->expectedQualities.size();
    if (numQualities == 2) {
        TestUtil::printSkipTest("Need more than 2 qualities to run this test");
    }

    const auto& expectedQuality = TestStream->expectedQualities[numQualities / 2];
    ASSERT_TRUE(expectedQuality.isValid()) << "Could not find expected quality: " << expectedQuality.name;

    const auto targetQuality = getQuality(expectedQuality.name);
    ASSERT_FALSE(targetQuality.name.empty()) << "Could not find target quality: " << expectedQuality.name;

    ASSERT_TRUE(m_qualityControlMonitor->waitFor(expectedQuality.name, [&]() {
        m_player->setQuality(targetQuality);
    },
        PlayerTestTimeout::qualityChange))
        << "Player timed out trying to switch to quality: " << expectedQuality.name;

    const auto& actual = m_player->getQuality();
    EXPECT_EQ(expectedQuality.height, actual.height);
    EXPECT_EQ(expectedQuality.width, actual.width);
}

#if defined(__ANDROID__)
TEST_F(PlayerQualityTest, DISABLED_SetQualityTwiceSwitchToLatter) // CVP-2091
#else
TEST_F(PlayerQualityTest, SetQualityTwiceSwitchToLatter)
#endif
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Could not load stream";

    const size_t numQualities = TestStream->expectedQualities.size();
    const auto& quality1 = TestStream->expectedQualities[numQualities - 1];
    const auto& quality2 = TestStream->expectedQualities[numQualities - 2];

    ASSERT_NE(quality1.name, quality2.name) << "Cannot run this test with only two qualities";

    const auto targetQuality1 = getQuality(quality1.name);
    ASSERT_FALSE(targetQuality1.name.empty()) << "Could not find target quality 1: " << quality1.name;
    const auto targetQuality2 = getQuality(quality2.name);
    ASSERT_FALSE(targetQuality2.name.empty()) << "Could not find target quality 2: " << quality2.name;

    ASSERT_TRUE(m_qualityControlMonitor->waitFor(quality2.name, [&]() {
        m_player->setQuality(targetQuality1);
        m_player->setQuality(targetQuality2);
    },
        PlayerTestTimeout::qualityChange))
        << "Player timed out trying to switch to quality: " << quality1.name;

    const auto& actual = m_player->getQuality();
    EXPECT_EQ(targetQuality2.height, actual.height);
    EXPECT_EQ(targetQuality2.width, actual.width);
}
