#pragma once
#include "player_test_base.hpp"
#include "player_test_timeout.hpp"
#include "state_change_monitor.hpp"
#include "test_util.hpp"
#include <functional>

using namespace twitch::test;
/**
 * Tests various Player::State transitions
 */
class PlayerStateTest : public PlayerTestBase, public testing::WithParamInterface<std::string> {
protected:
    void SetUp() override
    {
        PlayerTestBase::SetUp();
    }
};

TEST_F(PlayerStateTest, IdleOnInitialization)
{
    EXPECT_EQ(twitch::Player::State::Idle, m_player->getState());
}

TEST_F(PlayerStateTest, ReadyOnLoad)
{
    EXPECT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ready, [this]() {
        m_player->load(m_url);
    },
        PlayerTestTimeout::load));
}

TEST_F(PlayerStateTest, IdleOnStop)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ready, [this]() {
        m_player->load(m_url);
    },
        PlayerTestTimeout::load));

    EXPECT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Idle, [this]() {
        m_player->pause();
    },
        PlayerTestTimeout::stop));
}

TEST_F(PlayerStateTest, BufferOnStart)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ready, [this]() {
        m_player->load(m_url);
    },
        PlayerTestTimeout::load));

    EXPECT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Buffering, [this]() {
        m_player->play();
    },
        PlayerTestTimeout::start));
}

TEST_F(PlayerStateTest, PlayingOnStartAfterBuffer)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ready, [this]() {
        m_player->load(m_url);
    },
        PlayerTestTimeout::load));

    EXPECT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->play();
    },
        PlayerTestTimeout::start));
}

TEST_F(PlayerStateTest, PlayingAfterStop)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start));

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Idle, [this]() {
        m_player->pause();
    },
        PlayerTestTimeout::stop));

    EXPECT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->play();
    },
        PlayerTestTimeout::start))
        << "Player was not able to resume a paused stream";
}

/**
 * VOD specific tests for MediaPlayer::PlayerState
 */
class PlayerStateVODTest : public PlayerTestBase {
protected:
    void SetUp() override
    {
        PlayerTestBase::SetUp();
    }
};

TEST_F(PlayerStateVODTest, BufferOnSeekOutsideBuffer)
{
    if (testInput().isLive) {
        TestUtil::printSkipTest();
    }

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start));

    EXPECT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Buffering, [this]() {
        twitch::MediaTime offset(std::chrono::milliseconds(500));
        auto time = m_player->getBufferedPosition() + offset;
        m_player->seekTo(time);
    },
        PlayerTestTimeout::seek));
}

TEST_F(PlayerStateVODTest, EndedAtEndOfStream)
{
    if (testInput().isLive) {
        TestUtil::printSkipTest();
        return;
    }

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start));

    EXPECT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ended, [this]() {
        twitch::MediaTime offset(std::chrono::seconds(1));
        auto time = m_player->getDuration() - offset;
        m_player->seekTo(time);
    },
        PlayerTestTimeout::seek));
}
