#include "player_test_base.hpp"
#include "IntegrationTest.hpp"
#include "debug/ExceptionHandler.hpp"
#include "debug/trace.hpp"
#include "debug/TraceLog.hpp"
#include "../util/gtestenv.hpp"
#include <cassert>

using namespace twitch;
using namespace twitch::test;
using namespace testing;

PlayerTestBase::PlayerTestBase()
    : m_environment(IntegrationTest::environment)
{
    assert(m_environment);
}

void PlayerTestBase::SetUp()
{
    TraceLog::get().setLevel(IntegrationTest::environment->options().getLogLevel());

    m_player = GTestEnvironment::current->createPlayer(*this);
    m_stateMonitor = std::unique_ptr<PlayerStateChangeMonitor>(new PlayerStateChangeMonitor(m_player->getState()));
    m_callbackMonitor = std::unique_ptr<PlayerCallbackMonitor>(new PlayerCallbackMonitor());

    bool isLive = IntegrationTest::environment->testData().isLive;
    m_stateMachine = std::unique_ptr<PlayerStateMachine>(new PlayerStateMachine(m_player->getState(), isLive));

    m_url = IntegrationTest::environment->testData().url;
}

void PlayerTestBase::TearDown()
{
    m_player.reset();
    m_stateMonitor.reset();
    m_callbackMonitor.reset();
}

void PlayerTestBase::onError(const twitch::Error& error)
{
    m_log->debug("MediaPlayerTestBase::onError(): source=%s, result=%s, message=%s",
                 errorSourceString(error.source),
                 mediaResultString(error.result),
                 error.message.c_str());
    ADD_FAILURE();
}

void PlayerTestBase::onStateChanged(twitch::Player::State state)
{
    assert(m_stateMachine);
    auto previous = m_stateMachine->getCurrentState();
    EXPECT_TRUE(m_stateMachine->onStateChange(state)) << "Invalid player state transition from " << Player::stateToString(previous) << " to " << Player::stateToString(state);

    assert(m_stateMonitor);
    m_stateMonitor->onStateChanged(state);
}

void PlayerTestBase::onSeekCompleted(twitch::MediaTime time)
{
    assert(m_callbackMonitor);
    m_callbackMonitor->onSeekCompleted(time);
}
