#pragma once
#include "player_state_timer.hpp"
#include "player_test_base.hpp"
#include "player_test_timeout.hpp"
#include "time_util.hpp"
#include "timed_test_runner.hpp"
#include <cmath>
#include <functional>
#include <gtest/gtest.h>
#include <utility>
#include <thread>

/**
 * PlayerTimedTest validates anything related to time
 */
using namespace twitch::test;
class PlayerTimedTestBase : public PlayerTestBase {
public:
    void SetUp() override
    {
        PlayerTestBase::SetUp();
        m_stateTimer = std::unique_ptr<PlayerStateTimer>(new PlayerStateTimer(m_player->getState()));
    }

    void TearDown() override
    {
        m_stateTimer.reset();
        PlayerTestBase::TearDown();
    }

    void onStateChanged(twitch::Player::State state) override
    {
        PlayerTestBase::onStateChanged(state);
        if (m_stateTimer) {
            m_stateTimer->onStateChanged(state);
        }
    }

    testing::AssertionResult testTimeDoesNotChange(TimedTestRunner::time_t testTime)
    {
        (void)testTime;
        auto position = m_player->getPosition();
        if (m_testStartPosition == position) {
            return ::testing::AssertionSuccess();
        } else {
            return ::testing::AssertionFailure() << "Time changed from " << TimeUtil::toString<std::chrono::milliseconds>(m_testStartPosition.microseconds()) << " to " << TimeUtil::toString<std::chrono::milliseconds>(position.microseconds());
        }
    }

    testing::AssertionResult testTimeIsIncreasing(TimedTestRunner::time_t testTime)
    {
        (void)testTime;
        auto position = m_player->getPosition();
        if (m_testStartPosition < position) {
            m_testStartPosition = position;
            return ::testing::AssertionSuccess();
        } else {
            return ::testing::AssertionFailure() << "Time did not increase from " << TimeUtil::toString<std::chrono::milliseconds>(m_testStartPosition.microseconds());
        }
    }

protected:
    std::unique_ptr<PlayerStateTimer> m_stateTimer;

    twitch::MediaTime m_testStartPosition = twitch::MediaTime::zero();
};

class PlayerTimedTest : public PlayerTimedTestBase, public testing::WithParamInterface<std::string> {
public:
    void SetUp() override
    {
        PlayerTimedTestBase::SetUp();
    }
};

TEST_F(PlayerTimedTest, TimeDoesNotChangeOnLoad)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ready, [this]() {
        m_player->load(m_url);
    },
        PlayerTestTimeout::load))
        << "Player was not able to reach Ready state for test setup";

    TimedTestRunner runner(std::chrono::seconds(1), std::chrono::milliseconds(500));
    m_testStartPosition = m_player->getPosition();
    EXPECT_TRUE(runner.run(std::bind(&PlayerTimedTest::testTimeDoesNotChange, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(PlayerTimedTest, TimeIncreasesAfterStart)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was not able to reach Playing state for test setup";

#ifdef ENABLE_LOOPBACK
    //Loopback mode currently takes longer for first frame, hence the difference. To be revisited later (CVP-2494)
    TimedTestRunner runner(std::chrono::seconds(6), std::chrono::seconds(2));
#else
    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
#endif
    EXPECT_TRUE(runner.run(std::bind(&PlayerTimedTest::testTimeIsIncreasing, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));

}

TEST_F(PlayerTimedTest, TimeDoesNotChangeOnStop)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was not able to reach Playing state for test setup";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Idle, [this]() {
        m_player->pause();
    },
        PlayerTestTimeout::stop))
        << "Player was not able to reach Ready state for test setup";

    TimedTestRunner runner(std::chrono::seconds(1), std::chrono::milliseconds(500));
    m_testStartPosition = m_player->getPosition();
    EXPECT_TRUE(runner.run(std::bind(&PlayerTimedTest::testTimeDoesNotChange, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

class PlayerTimedVODTest : public PlayerTimedTestBase {
public:
    void SetUp() override
    {
        PlayerTimedTestBase::SetUp();

        // https://www.twitch.tv/lawgiver
        m_url = "https://www.twitch.tv/videos/164446609";       
    }
};

class PlayerTimedClipTest : public PlayerTimedTestBase {
public:
    void SetUp() override
    {
        PlayerTimedTestBase::SetUp();

        // 30 seconds or less clip
        m_url = "https://clips-media-assets2.twitch.tv/AT-cm%7C396612218.mp4";
    }
};

TEST_F(PlayerTimedVODTest, TimeIncreasesAfterSeekForward)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    twitch::MediaTime duration = m_player->getDuration();
    ASSERT_TRUE(m_callbackMonitor->waitForSeekCompleted([this, duration]() {
        m_player->seekTo(duration / 2);
    },
        PlayerTestTimeout::seek));

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&PlayerTimedTest::testTimeIsIncreasing, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

#ifdef ENABLE_LOOPBACK
TEST_F(PlayerTimedVODTest, DISABLED_TimeIncreasesAfterSeekWithinBuffer)
#else
TEST_F(PlayerTimedVODTest, TimeIncreasesAfterSeekWithinBuffer)
#endif
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    ASSERT_TRUE(m_callbackMonitor->waitForSeekCompleted([this]() {
        auto bufferLength = m_player->getBufferedPosition() - m_player->getPosition();
        m_player->seekTo(m_player->getPosition() + (bufferLength / 2));
    },
        PlayerTestTimeout::seek))
        << "Player could not resume Playing after seek";

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&PlayerTimedTest::testTimeIsIncreasing, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(PlayerTimedVODTest, TimeIncreasesAfterSeekOutsideBuffer)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Ready state for test setup";

    std::chrono::seconds testDuration(3);

    ASSERT_TRUE(m_callbackMonitor->waitForSeekCompleted([this, testDuration]() {
        auto targetSeekTime = m_player->getBufferedPosition() + std::chrono::seconds(30);
        EXPECT_LT(targetSeekTime + testDuration, m_player->getDuration()) << "Not enough time for test to seek and run timed test";
        m_player->seekTo(targetSeekTime);
    },
        PlayerTestTimeout::seek))
        << "Player could not resume Playing after seek";

    TimedTestRunner runner(testDuration, std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&PlayerTimedTest::testTimeIsIncreasing, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

#if PLAYERCORE_OS_IOS
TEST_F(PlayerTimedVODTest, DISABLED_TimeIncreasesAfterSeekBackward)
#else
TEST_F(PlayerTimedVODTest, TimeIncreasesAfterSeekBackward)
#endif
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    ASSERT_TRUE(m_callbackMonitor->waitForSeekCompleted([this]() {
        m_player->seekTo(m_player->getDuration() / 2);
    },
        PlayerTestTimeout::seek));

    ASSERT_TRUE(m_callbackMonitor->waitForSeekCompleted([this]() {
        m_player->seekTo(twitch::MediaTime::zero());
    },
        PlayerTestTimeout::seek));

    TimedTestRunner runner(std::chrono::seconds(3), std::chrono::seconds(1));
    EXPECT_TRUE(runner.run(std::bind(&PlayerTimedTest::testTimeIsIncreasing, this, std::placeholders::_1))) << "Test failed at " << TimeUtil::toString<std::chrono::milliseconds>((runner.getTotalTestTime()));
}

TEST_F(PlayerTimedClipTest, ClipSeekForwardInBufferWhenPaused)
{  
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    std::this_thread::sleep_for(std::chrono::seconds(10)); // give enough time for everything to be buffered

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Idle, [this]() {
        m_player->pause();
    }, PlayerTestTimeout::stop)) << "Player did not reach Idle state after calling pause";

    auto position = m_player->getPosition();
    auto seekForwardPosition = position + twitch::MediaTime(5.0f);

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this, seekForwardPosition]() {
        m_player->seekTo(seekForwardPosition);
        std::this_thread::sleep_for(std::chrono::seconds(2));
        m_player->play();
    }, PlayerTestTimeout::seek)) << "Player did not reach the correct state when seeking forward";
}

TEST_F(PlayerTimedClipTest, ClipSeekBackwardInBufferWhenPaused)
{    
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    std::this_thread::sleep_for(std::chrono::seconds(10)); // give enough time for everything to be buffered

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Idle, [this]() {
        m_player->pause();
    }, PlayerTestTimeout::stop)) << "Player did not reach Idle state after calling pause";

    auto position = m_player->getPosition();
    auto seekBackwardPosition = position - twitch::MediaTime(5.0f);

    // This test will need to be changed once VP-5236 is fixed
    // Note: This test probably will fail on the Web since VP-5236 doesn't impact Web. 
    ASSERT_TRUE(m_stateMonitor->waitFor({ twitch::Player::State::Buffering, twitch::Player::State::Idle, twitch::Player::State::Buffering, twitch::Player::State::Playing }, [this, seekBackwardPosition]() {
        m_player->seekTo(seekBackwardPosition);
        std::this_thread::sleep_for(std::chrono::seconds(2));
        m_player->play();
    }, PlayerTestTimeout::seek)) << "Player did not reach the correct state when seeking forward";
}

TEST_F(PlayerTimedClipTest, ClipSeekForwardInBufferWhenPlaying)
{   
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    std::this_thread::sleep_for(std::chrono::seconds(10)); // give enough time for everything to be buffered

    auto position = m_player->getPosition();
    auto seekForwardPosition = position + twitch::MediaTime(5.0f);

    ASSERT_TRUE(m_stateMonitor->waitWhileStayingInState(twitch::Player::State::Playing, [this, seekForwardPosition]() {
        m_player->seekTo(seekForwardPosition);
    }, std::chrono::seconds(2))) << "Player did not reach the correct state when seeking forward";
}

TEST_F(PlayerTimedClipTest, ClipSeekBackwardInBufferWhenPlaying)
{   
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->load(m_url);
        m_player->play();
    },
        PlayerTestTimeout::load + PlayerTestTimeout::start))
        << "Player was unable to reach Playing state for test setup";

    std::this_thread::sleep_for(std::chrono::seconds(10)); // give enough time for everything to be buffered

    auto position = m_player->getPosition();
    auto seekBackwardPosition = position - twitch::MediaTime(5.0f);

    // This test will need to be changed once VP-5236 is fixed
    // Note: This test probably will fail on the Web since VP-5236 doesn't impact Web. 
    ASSERT_TRUE(m_stateMonitor->waitFor({ twitch::Player::State::Buffering, twitch::Player::State::Playing }, [this, seekBackwardPosition]() {
        m_player->seekTo(seekBackwardPosition);
    }, PlayerTestTimeout::seek)) << "Player did not reach the correct state when seeking forward";
}


class PlayerTimedVODEndTest : public PlayerTimedTestBase {
public:
    void SetUp() override
    {
        PlayerTimedTestBase::SetUp();

        // https://www.twitch.tv/lawgiver
        // 10 seconds VOD
        m_url = "https://www.twitch.tv/videos/138540817";
    }

    const std::chrono::seconds endTimeout = std::chrono::seconds(15);
};

// CVP-1114 & CVP-2078
TEST_F(PlayerTimedVODEndTest, DISABLED_TimeAtEndIsAtEnd)
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ready, [this]() {
        m_player->load(m_url);
    },
        PlayerTestTimeout::load))
        << "Player was unable to reach Playing state for test setup";

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
        m_player->play();
    },
        PlayerTestTimeout::start));

    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ended, []() {}, PlayerTimedVODEndTest::endTimeout));

    twitch::MediaTime duration = m_player->getDuration();
    twitch::MediaTime position = m_player->getPosition();

    double durSecs = duration.seconds();
    double posSecs = position.seconds();

    ASSERT_LE(std::abs(posSecs - durSecs), 0.2f) << "VOD Ended state position is not equal to its duration";
}

class PlayerTimedPauseTest : public PlayerTimedTestBase {
public:
    void SetUp() override
    {
        PlayerTimedTestBase::SetUp();
        m_url = "https://fastly.vod.hls.ttvnw.net/478ae76b0cd41476ccef_twitch_26242922576_708852852/chunked/highlight-174321586.m3u8";
    }
    const std::chrono::seconds pauseTimeout = std::chrono::seconds(1);
};

#if defined(__ANDROID__)
TEST_F(PlayerTimedPauseTest, DISABLED_Pause) // CVP-2162
#else
TEST_F(PlayerTimedPauseTest, Pause)
#endif
{
    ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Ready, [this]() {
        m_player->load(m_url);
    },
        PlayerTestTimeout::load))
        << "Player was unable to reach Ready state";

    for (int i = 0; i < 3; ++i) {
        ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Playing, [this]() {
            m_player->play();
        },
            PlayerTestTimeout::start))
            << "Player was unable to reach Playing state";

        ASSERT_TRUE(m_stateMonitor->waitFor(twitch::Player::State::Idle, [this]() {
            m_player->pause();
        },
            PlayerTimedPauseTest::pauseTimeout))
            << "Player was unable to reach Idle state";
    }
}
