#include "hls/PlaylistParser.hpp"
#include <gtest/gtest.h>

using namespace twitch::hls;

TEST(Parser, Prefix)
{

    const std::string string1("this is my line");
    PlaylistParser parser(string1);
    parser.nextLine();
    ASSERT_TRUE(parser.readPrefix("this"));

    const std::string string2("Kappa123");
    PlaylistParser parser2(string2);
    parser2.nextLine();
    ASSERT_TRUE(parser2.readPrefix("Kappa"));
    ASSERT_STREQ("123", parser2.getLine().c_str());

    const std::string string3("SeemsGood");
    PlaylistParser parser3(string3);
    parser3.nextLine();
    ASSERT_FALSE(parser3.readPrefix("seems"));

    const std::string string4("");
    PlaylistParser parser4(string4);
    parser4.nextLine();
    ASSERT_TRUE(parser4.readPrefix(""));

    const std::string string5("FrankerZ");
    PlaylistParser parser5(string5);
    parser5.nextLine();
    ASSERT_TRUE(parser5.readPrefix(""));
}

TEST(Parser, AbsoluteURL)
{
    std::string base, relative;

    base = "https://www.twitch.tv/";
    relative = "https://www.twitch.tv/TrumpSC";
    ASSERT_EQ(relative, getAbsoluteURL(base, relative));

    base = "https://www.twitch.tv/";
    relative = "thekeefe/videos";
    ASSERT_EQ("https://www.twitch.tv/thekeefe/videos", getAbsoluteURL(base, relative));

    base = "https://www.twitch.tv/foo/nl_kripp";
    relative = "thekeefe/videos";
    ASSERT_EQ("https://www.twitch.tv/foo/thekeefe/videos", getAbsoluteURL(base, relative));

    base = "https://www.twitch.tv/foo/nl_kripp/";
    relative = "thekeefe/videos";
    ASSERT_EQ("https://www.twitch.tv/foo/nl_kripp/thekeefe/videos", getAbsoluteURL(base, relative));

    base = "https://www.twitch.tv/trumpsc/";
    relative = "/thekeefe/videos";
    ASSERT_EQ("https://www.twitch.tv/thekeefe/videos", getAbsoluteURL(base, relative));
}

TEST(Parser, AttributeList)
{
    std::string line = "key=\"value\",Kappa=123,quoted=\"im,a,string\"";
    PlaylistParser reader(line);
    reader.nextLine();
    std::map<std::string, std::string> attrs;

    ASSERT_TRUE(reader.parseAttributes(attrs));
    ASSERT_EQ(attrs["key"], "value");
    ASSERT_EQ(attrs["Kappa"], "123");
    ASSERT_EQ(attrs["quoted"], "im,a,string");
}
