#pragma once

#include "include/playercore/platform/HttpClient.hpp"

#include <memory>

class MockHttpClient : public twitch::HttpClient {
public:
    class MockHttpRequest : public twitch::HttpRequest {
    public:
        MockHttpRequest() = delete;
        MockHttpRequest(const std::string& url, twitch::HttpMethod method)
                : HttpRequest(url, method)
        {
        }
        void setHeader(const std::string& key, const std::string& value) override
        {
            m_headers[key] = value;
        }
        virtual void setContent(const std::vector<uint8_t>& content) override
        {
            m_content = content;
        }
        virtual void cancel() override {};

    private:
        twitch::HttpHeaderMap m_headers;
        std::vector<uint8_t> m_content;
    };
    class MockHttpResponse : public twitch::HttpResponse {
    public:
        MockHttpResponse() = delete;
        MockHttpResponse(const std::string& response, int status = 200)
            : HttpResponse(status)
            , m_response(response)
        {
        }

        virtual void read(ContentHandler onBuffer, ErrorHandler onError)
        {
            (void)onError;
            onBuffer(reinterpret_cast<const uint8_t*>(m_response.data()), m_response.size(), true);
        };

        std::string getHeader(const std::string& key) const override
        {
            auto iter = m_headers.find(key);
            return (iter == m_headers.end()) ? "" : iter->second;
        }

    private:
        std::string m_response;
        twitch::HttpHeaderMap m_headers;
    };

    using RequestSuccessHandler = std::function<std::shared_ptr<MockHttpResponse>(const twitch::HttpRequest&)>;
    using RequestErrorHandler = std::function<int(const twitch::HttpRequest&)>;

    class RequestProxy {
        friend class MockHttpClient;

    public:
        RequestProxy() = delete;

        RequestProxy& thenRespond(const RequestSuccessHandler& handler)
        {
            m_nextProxy = std::unique_ptr<RequestProxy>(new RequestProxy(handler));
            return *m_nextProxy;
        }

        RequestProxy& thenError(const RequestErrorHandler& handler)
        {
            m_nextProxy = std::unique_ptr<RequestProxy>(new RequestProxy(handler));
            return *m_nextProxy;
        }

        RequestProxy& thenRepeat(uint8_t times = 1)
        {
            if (times > std::numeric_limits<uint8_t>::max() - m_repeatCount) {
                m_repeatCount = std::numeric_limits<uint8_t>::max();
            } else {
                m_repeatCount += times;
            }

            return *this;
        }

    private:
        RequestProxy(const RequestSuccessHandler& handler)
            : m_requestSuccessHandler(handler)
            , m_repeatCount(0)
        {
        }
        RequestProxy(const RequestErrorHandler& handler)
            : m_requestErrorHandler(handler)
            , m_repeatCount(0)
        {
        }

        std::unique_ptr<RequestProxy> m_nextProxy;
        RequestSuccessHandler m_requestSuccessHandler;
        RequestErrorHandler m_requestErrorHandler;
        uint8_t m_repeatCount;
    };

    virtual std::shared_ptr<twitch::HttpRequest> createRequest(const std::string& url, twitch::HttpMethod method) override
    {
        return std::shared_ptr<twitch::HttpRequest>(new MockHttpRequest(url, method));
    }

    virtual void send(std::shared_ptr<twitch::HttpRequest> request, ResponseHandler onResponse, ErrorHandler onError)
    {
        if (m_nextProxy != nullptr) {
            if (m_nextProxy->m_requestSuccessHandler != nullptr) {
                std::shared_ptr<MockHttpResponse> response = m_nextProxy->m_requestSuccessHandler(*request);
                onResponse(response);
            } else if (m_nextProxy->m_requestErrorHandler != nullptr) {
                onError(m_nextProxy->m_requestErrorHandler(*request));
            }

            if (m_nextProxy->m_repeatCount == 0) {
                m_nextProxy = std::move(m_nextProxy->m_nextProxy);
            } else {
                m_nextProxy->m_repeatCount--;
            }
        }
    }

    RequestProxy& respond(const RequestSuccessHandler& handler)
    {
        m_nextProxy = std::unique_ptr<RequestProxy>(new RequestProxy(handler));
        return *m_nextProxy;
    }

    RequestProxy& error(const RequestErrorHandler& handler)
    {
        m_nextProxy = std::unique_ptr<RequestProxy>(new RequestProxy(handler));
        return *m_nextProxy;
    }

private:
    std::unique_ptr<RequestProxy> m_nextProxy;
};
