#include "util/Concurrency.hpp"
#include <gtest/gtest.h>
#include <thread>

using namespace twitch;

class ConcurrencyTest : public ::testing::Test {
protected:
    void TearDown() override
    {
        if (m_thread.joinable()) {
            m_thread.join();
        }
    }

    twitch::Mutex m_mutex;
    ConditionVariable m_condition;
    std::thread m_thread;
};

// See https://git-aws.internal.justin.tv/video/player-core/pull/2018/files/a759ce5210b7dcc4680074bd944085c513fe57da#diff-1648498acb11831ce62398b0cd424f89
TEST_F(ConcurrencyTest, DISABLED_NotifyBeforeTimeout)
{
    const std::chrono::milliseconds timeout(100);
    bool threadStarted = false;
    CvStatus waitReturn = CvStatus::no_timeout;

    // Start another thread that will wait on a condition variable
    m_thread = std::thread([&]() {
        {
            std::lock_guard<twitch::Mutex> lock(m_mutex);
            threadStarted = true;
            m_condition.notify_one();
        }

        std::unique_lock<twitch::Mutex> lock(m_mutex);
        waitReturn = m_condition.wait_for(lock, timeout);
        EXPECT_EQ(CvStatus::timeout, waitReturn);
    });

    {
        std::unique_lock<twitch::Mutex> lock(m_mutex);

        // Make sure the other thread has started
        ASSERT_TRUE(m_condition.wait_for(lock, std::chrono::milliseconds(500), [&]() { return threadStarted; }))
            << "Timed out waiting for the thread to start";
    }

    // Allow thread to reacquire lock to wait on condition
    std::this_thread::sleep_for(timeout / 4);

    {
        std::lock_guard<twitch::Mutex> lock(m_mutex);
        // Notify the condition variable before sleeping this thread
        m_condition.notify_one();
        std::this_thread::sleep_for(timeout);
    }

    // Allow the thread to join to check the condition variable status
    m_thread.join();

    EXPECT_EQ(CvStatus::timeout, waitReturn);
}

TEST_F(ConcurrencyTest, NotifyAfterTimeout)
{
    const std::chrono::milliseconds timeout(100);
    bool threadStarted = false;
    CvStatus waitReturn = CvStatus::no_timeout;

    // Start another thread that will wait on a condition variable
    m_thread = std::thread([&]() {
        {
            std::lock_guard<twitch::Mutex> lock(m_mutex);
            threadStarted = true;
            m_condition.notify_one();
        }

        std::unique_lock<twitch::Mutex> lock(m_mutex);
        waitReturn = m_condition.wait_for(lock, timeout);
        EXPECT_EQ(CvStatus::timeout, waitReturn);
    });

    {
        std::unique_lock<twitch::Mutex> lock(m_mutex);

        // Make sure the other thread has started
        ASSERT_TRUE(m_condition.wait_for(lock, std::chrono::milliseconds(500), [&]() { return threadStarted; }))
            << "Timed out waiting for the thread to start";
    }

    // Allow thread to reacquire lock to wait on condition
    std::this_thread::sleep_for(timeout / 4);

    {
        std::lock_guard<twitch::Mutex> lock(m_mutex);
        // Notify the condition variable after sleeping this thread
        std::this_thread::sleep_for(timeout);
        m_condition.notify_one();
    }

    // Allow the thread to join to check the condition variable status
    m_thread.join();

    EXPECT_EQ(CvStatus::timeout, waitReturn);
}
