#include "testenv.hpp"
#include "player/AsyncHttpClient.hpp"
#include <condition_variable>
#include <gtest/gtest.h>
#include <mutex>
#include <thread>
#include "test/common/util/gtestenv.hpp"

using namespace twitch;

// Factory methods for creating HttpClient objects
template <class T>
std::shared_ptr<HttpClient> CreateHttpClient();

template <>
std::shared_ptr<HttpClient> CreateHttpClient<HttpClient>()
{
    auto platform = GTestEnvironment::current->createPlatform();
    return platform->getHttpClient();
}

template <>
std::shared_ptr<HttpClient> CreateHttpClient<AsyncHttpClient>(
)
{
    static int iteration = 0;
    auto platform = GTestEnvironment::current->createPlatform();
    auto scheduler = platform->createScheduler("Scheduler(" + std::to_string(++iteration) + ")");
    return platform->createAsyncHttpClient(scheduler);
}

template <typename T>
class HttpTest : public ::testing::Test {
protected:
    std::shared_ptr<twitch::HttpClient> m_client;
    std::mutex m_mutex;
    std::condition_variable m_condition;
    int m_lastError = 0;
    bool m_done = false;

    HttpTest()
        : m_client(CreateHttpClient<T>())
    {
    }

    void SetUp() override
    {
        ASSERT_TRUE((bool)m_client);
    }

    void TearDown() override
    {
        EXPECT_EQ(0, m_lastError) << "Received unexpected error";

        m_client.reset();
    }

public:
    void onError(int error)
    {
        EXPECT_NE(error, 0);
        std::lock_guard<std::mutex> lock(m_mutex);
        m_lastError = error;
        m_done = true;
        m_condition.notify_one();
    }

    void onExpectedError(int error)
    {
        EXPECT_NE(error, 0);
        std::lock_guard<std::mutex> lock(m_mutex);
        m_done = true;
        m_condition.notify_one();
    }
};

// Register HttpClient types
#if !defined(ANDROID)
// On PS4 and iOS, we'll force usage of AsyncHttpClient since the client has been made mutex free and rely on AsyncHttp for proper locking behavior.
// We could allow for HttpClient to be callback/cancel safe as well but this would require adding a new mutex in PS4/iOS Http, so this would end up with two layers of mutexes when using it in an asynchttp way.
// Since we are always using AsyncHttpClient on PS4/iOS, we will not test the raw HttpClient implementation
typedef testing::Types<AsyncHttpClient> HttpImplementations;
#else
typedef testing::Types<HttpClient, AsyncHttpClient> HttpImplementations;
#endif
TYPED_TEST_CASE(HttpTest, HttpImplementations);

TYPED_TEST(HttpTest, GetValidURL)
{
    auto request = this->m_client->createRequest("https://www.twitch.tv", HttpMethod::GET);

    this->m_client->send(request,
        [&](std::shared_ptr<twitch::HttpResponse> response) {
            EXPECT_TRUE(response->isSuccess()) << "HttpResponse unexpectedly failed";
            response->read(
                [&](const uint8_t* buffer, size_t size, bool endOfStream) {
                    if (size > 0) {
                        EXPECT_TRUE(buffer != nullptr) << "Expected non-null buffer with size of " << size;
                    }

                    if (endOfStream) {
                        std::lock_guard<std::mutex> lock(this->m_mutex);
                        this->m_done = true;
                        this->m_condition.notify_one();
                    }
                    return true;
                },
                std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));
        },
        std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));

    std::unique_lock<std::mutex> lock(this->m_mutex);
    this->m_condition.wait(lock, [&]() { return this->m_done; });
}

TYPED_TEST(HttpTest, GetInvalidURL)
{
    auto request = this->m_client->createRequest("https://www.thisdomainshouldnotexist.com", HttpMethod::GET);

    this->m_client->send(request,
        [&](std::shared_ptr<twitch::HttpResponse> response) {
            EXPECT_FALSE(response->isSuccess()) << "HttpResponse unexpectedly succeeded";
        },
        std::bind(&HttpTest<TypeParam>::onExpectedError, this, std::placeholders::_1));

    std::unique_lock<std::mutex> lock(this->m_mutex);
    this->m_condition.wait(lock, [&]() { return this->m_done; });
}

// TODO: Disable until we have a server where we can test HTTP POST requests
TYPED_TEST(HttpTest, DISABLED_Post)
{
    std::string myContent = "myContent";
    // TODO: find a URL where we can make HTTP POST requests for testing
    auto request = this->m_client->createRequest("http://127.0.0.1:8000/post", HttpMethod::POST);
    request->setHeader("Content-Type", "text");
    std::vector<uint8_t> bytes(myContent.begin(), myContent.end());
    request->setContent(bytes);

    this->m_client->send(request,
        [&](std::shared_ptr<twitch::HttpResponse> response) {
            EXPECT_TRUE(response->isSuccess()) << "HttpResponse failed";
        },
        std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));

    std::unique_lock<std::mutex> lock(this->m_mutex);
    this->m_condition.wait(lock, [&]() { return this->m_done; });
}

TYPED_TEST(HttpTest, SendCanceledRequest)
{
    auto request = this->m_client->createRequest("https://www.twitch.tv", HttpMethod::GET);
    request->cancel();

    auto onResponse = [&](std::shared_ptr<twitch::HttpResponse>) {
        FAIL() << "Received unexpected response handler callback";
    };
    this->m_client->send(request, onResponse, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));
}

TYPED_TEST(HttpTest, CancelBeforeResponse)
{
    auto request = this->m_client->createRequest("https://www.twitch.tv", HttpMethod::GET);

    auto onResponse = [&](std::shared_ptr<twitch::HttpResponse>) {
        ADD_FAILURE() << "Received unexpected content handler callback";
    };

    this->m_client->send(request, onResponse, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));
    request->cancel();

    std::this_thread::sleep_for(std::chrono::seconds(1));
}

// This test makes sure that we are not allowed to cancel (from a different thread) while the `onResponse` callback
// is going on.
TYPED_TEST(HttpTest, CancelAndOnResponseAreAtomic)
{
    bool done = false;
    bool hasCanceled = false;

    auto request = this->m_client->createRequest("https://www.twitch.tv", HttpMethod::GET);
    auto onResponse = [&](std::shared_ptr<twitch::HttpResponse>) {
        {
            std::unique_lock<std::mutex> lock(this->m_mutex);
            done = true;
        }
        this->m_condition.notify_one();

        {
            std::unique_lock<std::mutex> lock(this->m_mutex);
            bool wasCanceled = this->m_condition.wait_for(lock, std::chrono::milliseconds(300), [&]() { return hasCanceled; });
            EXPECT_FALSE(wasCanceled) << "Not allowed to cancel during onResponse";
        }

        {
            std::unique_lock<std::mutex> lock(this->m_mutex);
            this->m_done = true;
        }
        this->m_condition.notify_one();
    };

    this->m_client->send(request, onResponse, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));

    // Wait to be in the `onResponse` callback because we want to cancel there
    {
        std::unique_lock<std::mutex> lock(this->m_mutex);
        bool succeeded = this->m_condition.wait_for(lock, std::chrono::seconds(5), [&]() { return done; });
        EXPECT_TRUE(succeeded) << "onResponse never set m_done to true";
    }

    // now cancel it
    {
        request->cancel(); // we expect this to block
        std::unique_lock<std::mutex> lock(this->m_mutex);
        hasCanceled = true;
    }
    this->m_condition.notify_one();

    {
        std::unique_lock<std::mutex> lock(this->m_mutex);
        EXPECT_TRUE(this->m_condition.wait_for(lock, std::chrono::seconds(1), [&]() { return this->m_done; }));
    }
}

// This test makes sure that we are not allowed to cancel (from a different thread) while the `onBuffer` callback
// is going on.
TYPED_TEST(HttpTest, CancelAndOnBufferAreAtomic)
{
    bool done = false;
    bool hasCanceled = false;

    auto request = this->m_client->createRequest("https://www.twitch.tv", HttpMethod::GET);
    auto onResponse = [&](std::shared_ptr<twitch::HttpResponse> response) {

        auto onBuffer = [&](const uint8_t*, size_t, bool) {
            {
                std::unique_lock<std::mutex> lock(this->m_mutex);
                done = true;
            }
            this->m_condition.notify_one();

            {
                std::unique_lock<std::mutex> lock(this->m_mutex);
                bool wasCanceled = this->m_condition.wait_for(lock, std::chrono::milliseconds(300), [&]() { return hasCanceled; });
                EXPECT_FALSE(wasCanceled) << "Not allowed to cancel during onResponse";
            }

            {
                std::unique_lock<std::mutex> lock(this->m_mutex);
                this->m_done = true;
            }
            this->m_condition.notify_one();
            return true;
        };

        response->read(onBuffer, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));
    };

    this->m_client->send(request, onResponse, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));

    // Wait to be in the `onResponse` callback because we want to cancel there
    {
        std::unique_lock<std::mutex> lock(this->m_mutex);
        bool succeeded = this->m_condition.wait_for(lock, std::chrono::seconds(5), [&]() { return done; });
        EXPECT_TRUE(succeeded) << "onResponse never set m_done to true";
    }

    // now cancel it
    {
        request->cancel(); // we expect this to block
        std::unique_lock<std::mutex> lock(this->m_mutex);
        hasCanceled = true;
    }
    this->m_condition.notify_one();

    {
        std::unique_lock<std::mutex> lock(this->m_mutex);
        EXPECT_TRUE(this->m_condition.wait_for(lock, std::chrono::seconds(1), [&]() { return this->m_done; }));
    }
}

TYPED_TEST(HttpTest, CancelRequestAfterResponse)
{
    auto request = this->m_client->createRequest("https://www.twitch.tv", HttpMethod::GET);

    auto onResponse = [&](std::shared_ptr<twitch::HttpResponse> response) {
        auto onBuffer = [](const uint8_t*, size_t, bool) {
            ADD_FAILURE() << "Received unexpected content handler callback";
            return true;
        };

        request->cancel();
        response->read(onBuffer, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));

        std::lock_guard<std::mutex> lock(this->m_mutex);
        this->m_done = true;
        this->m_condition.notify_one();
    };

    this->m_client->send(request, onResponse, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));

    std::unique_lock<std::mutex> lock(this->m_mutex);
    bool didComplete = this->m_condition.wait_for(lock, std::chrono::milliseconds(5000), [&]() { return this->m_done; });
    EXPECT_TRUE(didComplete);

    // If the test failed, we must cancel the request, otherwise we might hit dangling callbacks
    if (!didComplete) {
        request->cancel();
    }
}

TYPED_TEST(HttpTest, CancelRequestAfterResponseRead)
{
    auto request = this->m_client->createRequest("https://www.twitch.tv", HttpMethod::GET);

    size_t onBufferCalls = 0;

    auto onResponse = [&](std::shared_ptr<twitch::HttpResponse> response) {
        auto onBuffer = [&](const uint8_t*, size_t, bool) {
            {
                std::unique_lock<std::mutex> lock(this->m_mutex);
                ++onBufferCalls;
            }
            this->m_condition.notify_one();

            if (onBufferCalls == 1) {
                request->cancel();
                {
                    std::unique_lock<std::mutex> lock(this->m_mutex);
                    this->m_done = true;
                }
            } else {
                ADD_FAILURE() << "Received unexpected content handler callback";
            }
            return true;
        };

        response->read(onBuffer, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));
    };

    this->m_client->send(request, onResponse, std::bind(&HttpTest<TypeParam>::onError, this, std::placeholders::_1));

    std::unique_lock<std::mutex> lock(this->m_mutex);
    // Wait for the first onBuffer call before the request is cancelled
    EXPECT_TRUE(this->m_condition.wait_for(lock, std::chrono::milliseconds(5000), [&]() { return onBufferCalls == 1; }));

    // Allow the test to wait to ensure no additional onBuffer calls are made
    EXPECT_FALSE(this->m_condition.wait_for(lock, std::chrono::milliseconds(100), [&]() { return onBufferCalls > 1; }))
        << "onBuffer called after request was cancelled";

    // Ensure that cancel has returned
    EXPECT_TRUE(this->m_done) << "request->cancel() has not returned";
    if (!this->m_done) {
        request->cancel();
    }
}
